import base64
import logging
import os

from sandbox.common import utils
from sandbox.projects.common.decorators import retries

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


class MacKeychain(object):
    def __init__(self, name, path, password):
        self.name = name
        self.path = path
        self.password = password

    @retries(5, exceptions=(subprocess.CalledProcessError,))
    def unlock(self):
        subprocess.check_call([
            'security', 'unlock-keychain',
            '-p', self.password,
            self.name,
        ])

    def sign(self, filepath, identity):
        subprocess.check_call([
            'codesign',
            '-s', identity,
            '--keychain', self.path,
            filepath,
        ])


class MacKeychainEnvironment(SandboxEnvironment):
    DEFAULT_KEYCHAIN_PASSWORD_VAULT = 'MAC_SANDBOX_USER_PASSWORD'

    BROWSER_KEYCHAIN_PASSWORD_VAULT = 'BROWSER_MAC_DESKTOP_SIGNING_PASSWORD'
    BROWSER_KEYCHAIN_CONTENT_VAULT = 'BROWSER_MAC_DESKTOP_SIGNING_CONTENT'

    SAFE_STORAGE_SECRET_VAULT = 'MAC_SANDBOX_SAFESTORAGE_SECRET'

    def __init__(self, default_keychain_password_vault=DEFAULT_KEYCHAIN_PASSWORD_VAULT,
                 browser_keychain_password_vault=BROWSER_KEYCHAIN_PASSWORD_VAULT,
                 browser_keychain_content_vault=BROWSER_KEYCHAIN_CONTENT_VAULT,
                 safe_storage_secret_vault=SAFE_STORAGE_SECRET_VAULT):
        super(MacKeychainEnvironment, self).__init__()
        self.default_keychain_password_vault = default_keychain_password_vault
        self.browser_keychain_password_vault = browser_keychain_password_vault
        self.browser_keychain_content_vault = browser_keychain_content_vault
        self.safe_storage_secret_vault = safe_storage_secret_vault

    @classmethod
    def _keychains_dir(cls):
        return os.path.join(os.path.expanduser('~'), 'Library', 'Keychains')

    @utils.singleton_property
    def default_keychain(self):
        return MacKeychain(
            name='login.keychain',
            path=os.path.join(self._keychains_dir(), 'login.keychain-db'),
            password=sdk2.Vault.data(self.default_keychain_password_vault),
        )

    @utils.singleton_property
    def browser_keychain(self):
        return MacKeychain(
            name='desktop-signing.keychain',
            path=os.path.join(self._keychains_dir(), 'desktop-signing.keychain'),
            password=sdk2.Vault.data(self.browser_keychain_password_vault),
        )

    @utils.singleton_property
    def safe_storage_secret(self):
        return sdk2.Vault.data(self.safe_storage_secret_vault)

    def _reset_generic_password(self, account_name, service_name):
        subprocess.call([
            'security', 'delete-generic-password',
            '-s', service_name,
            self.default_keychain.name,
        ])
        subprocess.check_call([
            'security', 'add-generic-password',
            '-a', account_name,
            '-s', service_name,
            '-w', self.safe_storage_secret,
            '-A',
            self.default_keychain.name,
        ])
        subprocess.check_call([
            'security',
            'set-generic-password-partition-list',
            '-s', service_name,
            '-S', 'apple-tool:,unsigned:',
            '-k', self.default_keychain.password,
            self.default_keychain.name,
        ])

    def prepare(self):
        browser_keychain_content = base64.b64decode(sdk2.Vault.data(self.browser_keychain_content_vault))

        with open(self.browser_keychain.path, 'wb') as browser_keychain_file:
            browser_keychain_file.write(browser_keychain_content)

        subprocess.check_call([
            'security', 'list-keychains',
            '-s', self.default_keychain.path, self.browser_keychain.path,
        ])
        logging.debug('Keychains list:\n%s', subprocess.check_output(['security', 'list-keychains']))

        self.default_keychain.unlock()
        for account_name in ('Yandex', 'Chromium', 'Chrome'):
            self._reset_generic_password(account_name, '{} Safe Storage'.format(account_name))
