import contextlib
import logging
import os
import tarfile

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2


def _get_session_with_retries():
    session = requests.Session()
    retry = Retry(
        total=5,
        backoff_factor=0.3,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class SwiftToolchainEnvironment(SandboxEnvironment):
    BASE_URL = ('https://s3.mds.yandex.net/broinfra-tools/swift-toolchain-{platform}/'
                'swift-toolchain-{platform}-swift-{version}-release.yandex.{subrevision}.tar.gz')

    def __init__(self, platform, version, subrevision):
        super(SwiftToolchainEnvironment, self).__init__()
        self.platform = platform
        self.toolchain_archive_url = self.BASE_URL.format(platform=platform, version=version, subrevision=subrevision)

    @property
    def task(self):
        return sdk2.Task.current

    @property
    def extract_dir(self):
        return str(self.task.path('swift_toolchain'))

    def unpack_archive(self, archive_stream):
        os.makedirs(self.extract_dir)
        with tarfile.open(fileobj=archive_stream, mode='r|*') as archive:
            archive.extractall(self.extract_dir)

    @property
    def compiler_path(self):
        if self.platform == 'linux':
            toolchain_dir = self.extract_dir
        else:
            toolchain_dir = os.path.join(self.extract_dir, 'Library', 'Developer', 'Toolchains')
            toolchains = os.listdir(toolchain_dir)
            assert len(toolchains) == 1
            toolchain = toolchains[0]
            assert toolchain.endswith('.xctoolchain')
            toolchain_dir = os.path.join(toolchain_dir, toolchain)
        return os.path.join(toolchain_dir, 'usr', 'bin', 'swift')

    def prepare(self):
        session = _get_session_with_retries()
        logging.info('Downloading swift toolchain from %s', self.toolchain_archive_url)
        with contextlib.closing(session.get(
                self.toolchain_archive_url, timeout=(6, 12), stream=True)) as archive_response:
            archive_response.raise_for_status()
            self.unpack_archive(archive_response.raw)
        return self.compiler_path
