import itertools
import logging
import socket
import uuid

import requests

from sandbox.projects.common import decorators


@decorators.retries(5, delay=0.5, backoff=2, exceptions=(requests.RequestException, socket.error))
def run_teamcity_build(tc, branch, build_type, parameters=None, comment=None, tags=None, pool=None,
                       change=None, snapshot_dependencies=None, make_unique=True, make_deps_unique=False,
                       put_to_top=False):
    parameters = dict(parameters or {})

    if make_unique:
        parameters['unused_unique_parameter'] = str(uuid.uuid4())

    if make_deps_unique:
        parameters['reverse.dep.*.unused_unique_parameter'] = str(uuid.uuid4())

    build = tc.build_types[build_type].run_build(
        branch=branch,
        change=change,
        properties=parameters,
        comment=comment,
        tags=tags,
        snapshot_dependencies=snapshot_dependencies,
        pool=pool,
        put_to_top=put_to_top,
    )
    logging.info('Triggered build %s for build_type %s', build.web_url, build_type)
    logging.debug('Build parameters:\n%s', parameters)

    return build


def run_builds_with_deps(tc, branch, build_type, count, deps_count,
                         comment=None, change=None, parameters=None, tags=(), add_deps_tags=False,
                         put_to_top=False):
    builds = []

    # to limit dependant builds to `deps_count` we trigger `deps_count` builds
    # and re-use their deps later
    for _ in xrange(min(count, deps_count)):
        builds.append(run_teamcity_build(
            tc,
            branch=branch,
            build_type=build_type,
            change=change,
            parameters=parameters,
            comment=comment,
            tags=tags,
            make_unique=True,
            make_deps_unique=True,
            put_to_top=put_to_top,
        ))

    deps = [
        build.snapshot_dependencies for build in builds
    ]
    logging.debug('Snapshot dependencies for %s: %s', build_type, deps)

    # cycle through triggered deps for even distribution
    deps_iterator = itertools.cycle(deps)
    while len(builds) < count:
        builds.append(run_teamcity_build(
            tc,
            branch=branch,
            build_type=build_type,
            change=change,
            parameters=parameters,
            comment=comment,
            tags=tags,
            snapshot_dependencies=next(deps_iterator, None),
            make_unique=True,
            make_deps_unique=False,
            put_to_top=put_to_top,
        ))

    all_deps = [
        b for deps_pack in deps for b in deps_pack
    ]

    if tags and add_deps_tags:
        dep_build_ids = set(dep_build.id for build in builds for dep_build in build.snapshot_dependencies)
        for dep_build_id in dep_build_ids:
            tc.Build(id=dep_build_id).add_tags(tags)

    return builds, all_deps
