import errno
import functools
import logging
import os
import signal
import sys
import threading
import time

from sandbox import sdk2


class TimeoutError(RuntimeError):
    pass


def timeout(time_limit, exception_class=TimeoutError,
            exception_message='{fname} timeout'):

    class FunctionWithReturnValue(object):
        def __init__(self, function):
            self._function = function
            self._return_value = None
            self._exc_info = None

        def __call__(self, *args, **kwargs):
            try:
                self._return_value = self._function(*args, **kwargs)
            except:
                self._exc_info = sys.exc_info()

        @property
        def return_value(self):
            if self._exc_info:
                raise self._exc_info[0], self._exc_info[1], self._exc_info[2]
            else:
                return self._return_value

        def finalize(self):
            del self._exc_info
            self._exc_info = None

    def decorator(function):
        @functools.wraps(function)
        def wrapped(*args, **kwargs):
            function_rv = FunctionWithReturnValue(function)
            thread = threading.Thread(target=function_rv, args=args, kwargs=kwargs)
            thread.daemon = True
            thread.start()
            thread.join(time_limit)
            try:
                if thread.isAlive():
                    raise exception_class(exception_message.format(fname=function.__name__))
                else:
                    return function_rv.return_value
            finally:
                function_rv.finalize()
        return wrapped
    return decorator


class GracefulKillBeforeTimeoutMixin(object):
    """
    Mixin for sdk2.Task for gracefully killing processes in ProcessRegistry before timeout.
    Processes are killed in `GRACEFUL_KILL_SECONDS_BEFORE_TIMEOUT` with `GRACEFUL_KILL_SIGNAL`.
    On Windows, subprocesses should be attached to the same console but another process group.
    """
    GRACEFUL_KILL_SIGNAL = signal.CTRL_BREAK_EVENT if os.name == 'nt' else signal.SIGTERM
    GRACEFUL_KILL_SECONDS_BEFORE_TIMEOUT = 5 * 60

    def on_before_timeout(self, seconds):
        super(GracefulKillBeforeTimeoutMixin, self).on_before_timeout(seconds)
        if seconds > self.GRACEFUL_KILL_SECONDS_BEFORE_TIMEOUT:
            return

        for process in sdk2.helpers.ProcessRegistry:
            try:
                os.kill(process.pid, self.GRACEFUL_KILL_SIGNAL)
            except OSError as e:
                if e.errno != errno.ESRCH:
                    logging.exception('Failed to send signal %s to process %d', self.GRACEFUL_KILL_SIGNAL, process.pid)
            else:
                logging.info('Sent signal %s to process %d', self.GRACEFUL_KILL_SIGNAL, process.pid)

        time.sleep(seconds + 60)  # Sleep until timeout in order to reach TIMEOUT status.
