import contextlib
import json
import os
import shutil
import tempfile
import zipfile

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2


AVAILABLE_ARCHS = (X86, X64, ARM64) = ('x86', 'x64', 'arm64')


def _get_session_with_retries():
    import requests
    from requests.adapters import HTTPAdapter
    from requests.packages.urllib3.util.retry import Retry

    session = requests.Session()
    retry = Retry(
        total=5,
        backoff_factor=0.3,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class WindowsToolchainEnvironment(SandboxEnvironment):
    BASE_URL = 'https://s3.mds.yandex.net/broinfra-tools/win-toolchain/{}.zip'

    def __init__(self, version, arch=X64):
        assert arch in AVAILABLE_ARCHS
        super(WindowsToolchainEnvironment, self).__init__()
        self.arch = arch
        self.version = version
        self.win_toolchain_url = self.BASE_URL.format(version)

    @property
    def task(self):
        return sdk2.Task.current

    @property
    def extract_dir(self):
        return str(self.task.path('win_toolchain_{}'.format(self.version)))

    def unpack_archive(self, archive_stream):
        os.makedirs(self.extract_dir)
        with tempfile.TemporaryFile() as archive_file:
            shutil.copyfileobj(archive_stream, archive_file)
            archive_file.seek(0)
            with zipfile.ZipFile(archive_file, 'r', zipfile.ZIP_DEFLATED, True) as archive:
                archive.extractall(self.extract_dir)

    def find_win_sdk_dir(self):
        from natsort import natsorted
        sdks_dir = os.path.join(self.extract_dir, 'Windows Kits')
        available_sdks = os.listdir(sdks_dir)
        assert len(available_sdks) > 0
        # Use latest available sdk
        latest_sdk = natsorted(available_sdks)[-1]
        return os.path.join(sdks_dir, latest_sdk)

    def _set_environment(self):
        win_sdk_dir = self.find_win_sdk_dir()
        environment_file_path = os.path.join(win_sdk_dir, 'bin', 'SetEnv.{}.json'.format(self.arch))
        with open(environment_file_path) as environment_file:
            environment = json.load(environment_file)

        for env_name, values in environment['env'].items():
            paths = [os.path.join(self.extract_dir, *parts) for parts in values]
            # The only used compiler that can use win toolchain on non-Windows is clang-cl that expects paths to
            # be ;-separated. the only special case here is PATH.
            is_path = env_name == 'PATH'
            separator = os.pathsep if env_name == 'PATH' else ';'
            value = separator.join(paths)
            if is_path:
                os.environ[env_name] = os.environ[env_name] + separator + value
            else:
                os.environ[env_name] = value

    def prepare(self):
        session = _get_session_with_retries()
        with contextlib.closing(session.get(self.win_toolchain_url, timeout=(6, 12), stream=True)) as archive_response:
            archive_response.raise_for_status()
            self.unpack_archive(archive_response.raw)

        self._set_environment()
        return self.extract_dir
