import logging
import re

import sandbox
from sandbox import sdk2
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin, DEFAULT_LXC_CONTAINER_RESOURCE_ID
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.conflict_score.BrowserReportPrConflictScore import utils
import sandbox.common.types.client as ctc


BRANCH_REGEX = re.compile(r'refs/heads/(.+)')
PULL_REQUEST_ID_REGEX = re.compile(r'refs/pull-requests/(\d+)/merge-pin')


class BrowserReportPrConflictScore(LinuxBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        container_resource = DEFAULT_LXC_CONTAINER_RESOURCE_ID
        client_tags = ctc.Tag.BROWSER
        environments = (
            GitEnvironment('2.32.0'),
        )
        cores = 2
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):
        # Default value is a secret of robot-stash.
        # This secret has a token for app browser-conflict-score, which has access to Bitbucket and Teamcity API.
        bitbucket_token_secret = sdk2.parameters.YavSecret(
            'YAV secret identifier for access to Bitbucket API and Teamcity API',
            default='sec-01cjh86k7rywbrbz7xqzdabhv7#browser-conflict-score-token'
        )
        commit = sdk2.parameters.String('Merge-pin commit', required=True)
        branch = sdk2.parameters.String('PR branch', required=True)
        merge_pin_commit_to_get_conflict_score = sdk2.parameters.Bool(
            'Use merge pin commit to find from- and target-branches', default=True
        )
        approvers = sdk2.parameters.List('Approvers', required=True, default=['dmgor'])

    class Context(sdk2.Task.Context):
        # Extra stats that will be reported as Teamcity build statistics by plugin.
        teamcity_build_statistics = {}

    @property
    @sandbox.common.utils.singleton
    def bitbucket_client(self):
        from bitbucket import BitBucket
        token = self.Parameters.bitbucket_token_secret.data()[self.Parameters.bitbucket_token_secret.default_key]
        bb_client = BitBucket('https://bitbucket.browser.yandex-team.ru', token=token)
        return bb_client

    def get_from_and_target_pr_branches(self, pr, bb_client):
        if self.Parameters.merge_pin_commit_to_get_conflict_score:
            # Get info via merge-pin commit (more suitable for BitBucket 7).
            merge_pin_commit = bb_client.projects['STARDUST'].repos['browser'].commits[self.Parameters.commit]
            logging.debug('merge_pin_commit == %s', merge_pin_commit)

            assert len(merge_pin_commit.parents) == 2, 'Merge-pin commit must have exactly 2 parents'

            from_pr_branch = str(merge_pin_commit.parents[1].id)  # equal to from_pr_branch_name
            logging.debug('from_pr_branch_head == %s', from_pr_branch)

            target_pr_branch = str(merge_pin_commit.parents[0].id)
            logging.debug('target_pr_branch_commit == %s', target_pr_branch)
        else:
            # Get info via branches names (reflects actual changes)
            from_pr_branch = str(BRANCH_REGEX.findall(pr.from_ref.id)[0])
            logging.debug('from_pr_branch_name == %s', from_pr_branch)

            target_pr_branch = str(BRANCH_REGEX.findall(pr.to_ref.id)[0])
            logging.debug('target_pr_branch_name == %s', target_pr_branch)

        return from_pr_branch, target_pr_branch

    def publish_statistics_to_teamcity(self, **kwargs):
        for statistics, value in kwargs.items():
            self.Context.teamcity_build_statistics[statistics] = value

    def on_execute(self):
        from browser.infra.library.conflict_score.conflict_score import get_pr_full_conflict_score_info

        bb_client = self.bitbucket_client

        # Get pull-request.
        pull_request_id = PULL_REQUEST_ID_REGEX.findall(str(self.Parameters.branch))[0]
        logging.debug('pull_request_id == %s', pull_request_id)
        pr = bb_client.projects['STARDUST'].repos['browser'].pull_requests[pull_request_id]
        logging.debug('pr == %s', pr)

        # Get its from-branch and target-branch.
        from_pr_branch, target_pr_branch = self.get_from_and_target_pr_branches(pr=pr, bb_client=bb_client)

        # Update repo.
        bb_repo = repositories.Stardust.browser()
        bb_repo.update_cache_repo(from_pr_branch, target_pr_branch)

        # Get conflict score info.
        repo_root_path = bb_repo.cache_repo_dir
        logging.debug('repo_root_path == %s', repo_root_path)
        full_conflict_score_info = get_pr_full_conflict_score_info(
            from_pr_branch=from_pr_branch,
            target_pr_branch=target_pr_branch,
            repo_root_path=repo_root_path
        )
        logging.debug('full_conflict_score_info == %s', full_conflict_score_info)

        # Add conflict score comment to PR if needed.
        conflict_score_change_comment_optional = utils.maybe_add_new_comment(
            pr=pr,
            previous_total_conflict_score=full_conflict_score_info.from_conflict_score,
            current_total_conflict_score=full_conflict_score_info.target_conflict_score,
            total_conflict_score_change=full_conflict_score_info.total_conflict_score_change
        )

        # Add too-many-files comment if needed.
        annotations = utils.get_annotations_by_file(
            conflict_score_change_by_file=full_conflict_score_info.conflict_score_change_by_file
        )
        utils.maybe_add_too_many_conflicting_files_comment(
            pr=pr, annotations=annotations, parent_comment=conflict_score_change_comment_optional
        )

        utils.update_code_insights_report(
            pr=pr,
            total_conflict_score_change=full_conflict_score_info.total_conflict_score_change,
            annotations=annotations
        )

        utils.manage_conflict_score_veto(
            pr=pr,
            total_conflict_score_change=full_conflict_score_info.total_conflict_score_change,
            approvers=self.Parameters.approvers
        )

        self.publish_statistics_to_teamcity(
            conflict_score_change=full_conflict_score_info.total_conflict_score_change
        )

        logging.info('Added code-insights')
