import logging
import re


MAX_NUM_ANNOTATIONS = 1000  # BYIN-14273
VETO_THRESHOLD = 30


# If you want to change comment format, change both variables, first for rendering, second for parsing.
COMMENT_TEXT_FORMAT_PATTERN = 'Total conflict_score: `{total:+d} ({previous} -> {current})`'
COMMENT_TEXT_REGEX_PATTERN = re.compile(r'^Total conflict_score: `([+-]?\d+) \((\d+) -> (\d+)\)`$')


def render_pr_conflict_score_comment(
    total_conflict_score_change, previous_total_conflict_score, current_total_conflict_score
):
    """
    :type total_conflict_score_change: int
    :type previous_total_conflict_score: int
    :type current_total_conflict_score: int
    :rtype: str
    """
    return COMMENT_TEXT_FORMAT_PATTERN.format(
        total=total_conflict_score_change,
        previous=previous_total_conflict_score,
        current=current_total_conflict_score
    )


def parse_pr_conflict_score_comment(comment_text):
    """
    Parse conflict score info from comment text, else return None.

    Comment with conflict score info looks like 'Total conflict_score: `+29 (380 -> 409)`'.
    If comment looks differently, return None.
    conflict_score_change_by_file in returned PrConflictScoreInfo equals to None.

    :type comment_text: str
    :rtype: PrConflictScoreInfo | None
    """
    from browser.infra.library.conflict_score.conflict_score import PrConflictScoreInfo

    match = COMMENT_TEXT_REGEX_PATTERN.match(comment_text)
    if match is None:
        # Comment text is not a conflict score change comment.
        return None

    total_conflict_score_change = int(match.group(1))
    from_conflict_score = int(match.group(2))
    target_conflict_score = int(match.group(3))

    return PrConflictScoreInfo(
        total_conflict_score_change=total_conflict_score_change,
        from_conflict_score=from_conflict_score,
        target_conflict_score=target_conflict_score,
        conflict_score_change_by_file=None
    )


def get_previous_pr_conflict_score_info(pull_request):
    """
    Find most recent conflict score info comment for pull_request, return conflict score info from it. Else return None.

    :param pull_request: PullRequestFactory object
    :return: Any | None
    """
    comment_activities = [a for a in pull_request.activities if a.action == 'COMMENTED']
    for activity in comment_activities:
        comment_text = activity.comment.text
        logging.debug('comment_text: %s', comment_text)
        pr_conflict_score_info = parse_pr_conflict_score_comment(comment_text)
        if pr_conflict_score_info is None:
            continue
        logging.debug('Return PR conflict score info')
        return pr_conflict_score_info
    logging.debug('No conflict score comments, return None')
    return None


def maybe_add_new_comment(
    pr, previous_total_conflict_score, current_total_conflict_score, total_conflict_score_change,
):
    """
    Create new conflict score change comment if needed, then return it.

    If last conflict score comment has same parameters, do not create a new comment, return None.
    Else create new comment with conflict score change info and return it.
    :param pr: PullRequestFactory object
    :type previous_total_conflict_score: int
    :type current_total_conflict_score: int
    :type total_conflict_score_change: int
    :return: New created comment or None
    :rtype: Any | None
    """
    from browser.infra.library.conflict_score.conflict_score import PrConflictScoreInfo

    previous_pr_conflict_score_info = get_previous_pr_conflict_score_info(pr)
    logging.debug('previous_pr_conflict_score_info == %s', previous_pr_conflict_score_info)
    if previous_pr_conflict_score_info is None and total_conflict_score_change == 0:
        logging.debug('Comment was not added because conflict score change is 0, no previous comments')
        return None

    current_pr_conflict_score_info = PrConflictScoreInfo(
        total_conflict_score_change=total_conflict_score_change,
        from_conflict_score=previous_total_conflict_score,
        target_conflict_score=current_total_conflict_score,
        conflict_score_change_by_file=None
    )
    logging.debug('current_pr_conflict_score_info == %s', current_pr_conflict_score_info)
    if (
        previous_pr_conflict_score_info is not None
        and previous_pr_conflict_score_info == current_pr_conflict_score_info
    ):
        logging.debug('Comment was not added because conflict score info is same as in previous such comment')
        return None

    comment_text = render_pr_conflict_score_comment(
        total_conflict_score_change=total_conflict_score_change,
        previous_total_conflict_score=previous_total_conflict_score,
        current_total_conflict_score=current_total_conflict_score
    )
    comment = pr.comments.create_comment(comment_text, parent_comment=None)
    logging.info('Added comment #%s "%s"', comment.id, comment.text)
    return comment


def manage_conflict_score_veto(
    pr, total_conflict_score_change, approvers
):
    """
    Add or delete conflict score veto for pr relying on total_conflict_score_change and previous veto.

    Add veto if total_conflict_score_change is too large and there is no veto yet.
    Delete veto total_conflict_score_change is appropriate and there is such veto.

    :param pr: PullRequestFactory
    :type total_conflict_score_change: int
    :type approvers: list[str]
    """
    vetoes = pr.vetos
    if total_conflict_score_change >= VETO_THRESHOLD:
        if vetoes.with_slug('conflict_score') is None:
            vetoes.add(
                slug='conflict_score',
                summary='Too high conflict score change',
                details=(
                    'This Pull Request increases the conflict score too high (threshold is {threshold}).'
                    ' Add one of veto-approvers ({veto_approvers}) to review this PR'
                ).format(
                    threshold=VETO_THRESHOLD,
                    veto_approvers=', '.join(approvers)
                ),
                approvers=approvers
            )
            logging.debug('Added conflict_score veto')
        else:
            logging.debug('conflict_score veto was already added earlier')
    else:
        if vetoes.with_slug('conflict_score') is not None:
            vetoes.delete(slug='conflict_score')
            logging.debug('Removed conflict_score veto')
        else:
            logging.debug('There was no conflict_score veto')


def get_annotations_by_file(conflict_score_change_by_file):
    """
    :type conflict_score_change_by_file: dict[str, int]
    :rtype: list[CodeInsightsAnnotationItem]
    """
    from bitbucket.plugins.code_insights import CodeInsightsAnnotationItem, IssueSeverity, IssueType

    annotations = []
    for file_rel_path, file_conflict_score in conflict_score_change_by_file.items():
        if file_conflict_score == 0:
            continue
        annotations.append(
            CodeInsightsAnnotationItem(
                path=file_rel_path,
                line=0,
                severity=IssueSeverity.MEDIUM,
                message='Conflict score: {:+d}'.format(file_conflict_score),
                type=IssueType.CODE_SMELL,
            )
        )
    return annotations


def get_conflict_score_report_data(total_conflict_score_change):
    """
    :type total_conflict_score_change: int
    :rtype: list[ReportData]
    """
    from bitbucket.plugins.code_insights import ReportData, ValueType

    return [
        ReportData(
            title='Conflict score:',
            type=ValueType.TEXT, value='{:+d}'.format(total_conflict_score_change)
        ),
    ]


def get_conflict_score_report_result(total_conflict_score_change):
    """
    :type total_conflict_score_change: int
    :rtype: ReportResult
    """
    from bitbucket.plugins.code_insights import ReportResult

    return (
        ReportResult.FAIL
        if total_conflict_score_change >= VETO_THRESHOLD
        else ReportResult.PASS
    )


def update_code_insights_report(
    pr, total_conflict_score_change, annotations
):
    """
    :param pr: PullRequestFactory object
    :type total_conflict_score_change: int
    :type annotations: list[CodeInsightsAnnotationItem]
    """
    if not annotations:
        logging.debug("Didn't add any annotations because no conflict score changed")
        return

    latest_commit = pr.from_ref.latest_commit
    report = latest_commit.code_insights_reports.create_report(
        key='conflict_score',
        title='Conflict score report',
        result=get_conflict_score_report_result(total_conflict_score_change),
        data=get_conflict_score_report_data(total_conflict_score_change),
        reporter='Conflict score calculator'
    )
    report.annotations.delete()

    if len(annotations) > MAX_NUM_ANNOTATIONS:  # BYIN-14273
        annotations = annotations[:MAX_NUM_ANNOTATIONS]

    report.annotations.add(annotations)
    logging.debug('Added conflict score annotations')


def maybe_add_too_many_conflicting_files_comment(pr, annotations, parent_comment):
    """
    Create new conflict score change comment if needed, then return it.

    If there is no parent_comment, or number of files does not exceed MAX_NUM_ANNOTATIONS,
    do not create new comment, return None.

    :param pr: PullRequestFactory object
    :type annotations: list[CodeInsightsAnnotationItem]
    :type parent_comment: Any | None
    :rtype: Any | None
    """
    if parent_comment is None:
        logging.debug('Comment was not added because there is no conflict score')
        return None
    if len(annotations) <= MAX_NUM_ANNOTATIONS:  # BYIN-14273
        logging.debug('Comment was not added because there number of files does not exceed MAX_NUM_ANNOTATIONS')
        return None

    logging.debug(
        'Added conflict score annotations for %d files only because of code-insights API limit',
        MAX_NUM_ANNOTATIONS
    )
    comment = pr.comments.create_comment(
        'Added conflict score annotations for {} files only because of code-insights API limit'.format(
            MAX_NUM_ANNOTATIONS
        ),
        parent_comment=parent_comment
    )
    return comment
