import collections
import datetime
from dateutil import tz
from email.mime.text import MIMEText
import jinja2
import logging
import requests
import smtplib

import sandbox
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin, DEFAULT_LXC_CONTAINER_RESOURCE_ID
from sandbox.projects.browser.common.git import GitEnvironment, repositories


BITBUCKET_SERVER_URL = 'https://bitbucket.browser.yandex-team.ru'
STARDUST_PROJECT = 'STARDUST'
BROWSER_REPOSITORY = 'browser'
TEAMCITY_SERVER_URL = 'https://teamcity.browser.yandex-team.ru'
TEAMCITY_BUILD_TYPE_ID = 'Browser_Tests_ReportPrConflictScore'

CONFLICT_SCORE_CHANGE_STATISTICS_ID = 'sandbox.conflict_score_change'

EMAIL_SENDER = 'robot-browser-infra@yandex-team.ru'
SMTP_URL = 'yabacks.yandex.ru'
SMTP_PORT = 25


PullRequestInfoForReport = collections.namedtuple(
    'PullRequestInfoForReport',
    [
        'id',
        'url',
        'people',
        'conflict_score_change',
        'formatted_conflict_score_change'
    ]
)


def format_score(score):
    """
    :type score: int | None
    :return: html string. If score is None, use em-dash instead of number.
    :rtype: str
    """
    if isinstance(score, int):
        score_str = '{:+d}'.format(score)
        if score >= 30:
            return '<font color="red"><b>{}</b></font>'.format(score_str)
        if score >= 10:
            return '<font color="red">{}</font>'.format(score_str)
        if score > 0:
            return '<font color="darkred">{}</font>'.format(score_str)
        if score < 0:
            return '<font color="green">{}</font>'.format(score_str)
    if score is None:
        return '<font color="black"><b>&mdash;</b></font>'
    return '<font color="grey">{}</font>'.format(score)


class BrowserSendConflictScoreReport(LinuxBinaryTaskMixin, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        container_resource = DEFAULT_LXC_CONTAINER_RESOURCE_ID
        client_tags = ctc.Tag.BROWSER
        environments = (
            GitEnvironment('2.32.0'),
        )
        cores = 2
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):
        # Default value is a secret of robot-stash.
        # This secret has a token for app browser-conflict-score, which has access to Bitbucket and Teamcity API.
        bitbucket_token_secret = sdk2.parameters.YavSecret(
            'YAV secret identifier for access to Bitbucket API and Teamcity API',
            default='sec-01cjh86k7rywbrbz7xqzdabhv7#browser-conflict-score-token'
        )
        days = sdk2.parameters.Integer(
            'Maximum age of PR merging to include in this report (in days)', default=7, required=True
        )
        branch = sdk2.parameters.String(
            'PR merged into this branch will be reported', default='master', required=True
        )
        recipients = sdk2.parameters.List(
            'Emails to send report to',
            default=[
                'golubtsov@yandex-team.ru',
                'browser-infra-bots@yandex-team.ru',
                'dmgor@yandex-team.ru',
                'browser-merge@yandex-team.ru',
            ],
            required=True
        )

    @property
    @sandbox.common.utils.singleton
    def bitbucket_client(self):
        from bitbucket import BitBucket
        token = self.Parameters.bitbucket_token_secret.data()[self.Parameters.bitbucket_token_secret.default_key]
        bb_client = BitBucket(BITBUCKET_SERVER_URL, token=token)
        return bb_client

    @property
    @sandbox.common.utils.singleton
    def teamcity_client(self):
        from teamcity_client.client import TeamcityClient
        token = self.Parameters.bitbucket_token_secret.data()[self.Parameters.bitbucket_token_secret.default_key]
        tc_client = TeamcityClient(server_url=TEAMCITY_SERVER_URL, auth=token)
        return tc_client

    @property
    @sandbox.common.utils.singleton
    def bitbucket_repo(self):
        return self.bitbucket_client.projects[STARDUST_PROJECT].repos[BROWSER_REPOSITORY]

    @staticmethod
    def get_pull_request_url(pull_request_id):
        """
        :type pull_request_id: int | str
        :return: Relative url to pull_request for Bitbucket REST API
        :rtype: str
        """
        return '{}/projects/{}/repos/{}/pull-requests/{}/overview'.format(
            BITBUCKET_SERVER_URL,
            STARDUST_PROJECT,
            BROWSER_REPOSITORY,
            pull_request_id
        )

    @classmethod
    def get_pull_request_info_for_report(cls, pr, conflict_score_change):
        """
        Return all data of pull_request to be included or rendered into html of email message.

        :param pr: PullRequestFactory object
        :type conflict_score_change: int | None
        :rtype: PullRequestInfoForReport
        """
        return PullRequestInfoForReport(
            id=pr.id,
            url=cls.get_pull_request_url(pr.id),
            people=[pr.author.user.name] + sorted(r.user.name for r in pr.reviewers),
            conflict_score_change=conflict_score_change,
            formatted_conflict_score_change=format_score(conflict_score_change)
        )

    @staticmethod
    def count_conflict_score_on_the_fly(pr_merge_commit):
        """
        Count conflict score of PR by its merge commit.

        If pr has no bound report_pr_conflict_score Teamcity build, this method is called.
        This method counts conflict score change based on PR merge commit of pr and returns it.

        :param pr_merge_commit: CommitFactory object
        :rtype: int
        """
        from browser.infra.library.conflict_score.conflict_score import get_pr_full_conflict_score_info

        # Get info via PR merge commit (more suitable for BitBucket 7, we cannot rely on merge-pin commit anymore).
        logging.debug('pr_merge_commit == %s', pr_merge_commit)

        assert len(pr_merge_commit.parents) == 2, 'PR merge commit must have exactly 2 parents'

        from_pr_branch_head = str(pr_merge_commit.parents[1].id)  # equal to from_pr_branch_name
        target_pr_branch_head = str(pr_merge_commit.parents[0].id)

        logging.debug('from_pr_branch_head == %s', from_pr_branch_head)
        logging.debug('target_pr_branch_head == %s', target_pr_branch_head)

        bb_repo = repositories.Stardust.browser()
        bb_repo.update_cache_repo(from_pr_branch_head, target_pr_branch_head)
        repo_root_path = bb_repo.cache_repo_dir

        pr_full_conflict_score_info = get_pr_full_conflict_score_info(
            from_pr_branch=from_pr_branch_head,
            target_pr_branch=target_pr_branch_head,
            repo_root_path=repo_root_path
        )
        logging.debug('pr_full_conflict_score_info == %s', str(pr_full_conflict_score_info))
        logging.debug('conflict_score_change == %d', int(pr_full_conflict_score_info.total_conflict_score_change))

        return int(pr_full_conflict_score_info.total_conflict_score_change)

    def get_conflict_score_change(self, pr, pr_merge_commit):
        """
        If pr has sandbox.conflict_score_change statistics, return its value.
        Else if pr has merge-pin commit but no report_pr_conflict_score Teamcity build,
        return conflict score change and return this value.
        Else assume something is missing or corrupted, so pr should be skipped. Return None.

        :param pr: PullRequestFactory object
        :param pr_merge_commit: CommitFactory object
        :rtype: int | None
        """
        logging.debug('pr.id == %d', pr.id)

        if pr.references.merge_pin is None:
            # BYIN-14262: we include such PRs in report, but assume they do not affect conflict score
            logging.debug('pr #%d: no merge-pin commit, or merge-pin commit is corrupted, so skip it')
            return None

        if pr.merge_pin_teamcity_builds is None:
            # BYIN-14262: we include such PRs in report, but assume they do not affect conflict score
            logging.debug('pr #%d has no merge-pin commit builds, so skip it', pr.id)
            return None

        logging.debug('pr #%d has merge-pin commit builds', pr.id)

        if TEAMCITY_BUILD_TYPE_ID in pr.merge_pin_teamcity_builds:
            teamcity_conflict_score_id = pr.merge_pin_teamcity_builds[TEAMCITY_BUILD_TYPE_ID].id

            tc_build = self.teamcity_client.Build(id=teamcity_conflict_score_id)
            try:
                statistics = tc_build.statistics
            except requests.HTTPError:
                logging.debug(
                    'Last merge-pin commit for this PR does not have any required Teamcity build.'
                    ' Probably the pull request was merged before launching Teamcity builds'
                )
            else:
                logging.debug(
                    'pr #%d has build %s with id %d', pr.id, TEAMCITY_BUILD_TYPE_ID, teamcity_conflict_score_id
                )
                if CONFLICT_SCORE_CHANGE_STATISTICS_ID in statistics:
                    conflict_score_change = int(tc_build.statistics[CONFLICT_SCORE_CHANGE_STATISTICS_ID])
                    logging.debug('Found sandbox.conflict_score_change statistics: %d', conflict_score_change)
                    return conflict_score_change
                else:
                    logging.debug(
                        'sandbox.conflict_score_change statistics not found, build may be corrupted, so skip it'
                    )
                    return None

        logging.debug('pr #%d has NO build with build type id %s', pr.id, TEAMCITY_BUILD_TYPE_ID)
        return self.count_conflict_score_on_the_fly(pr_merge_commit=pr_merge_commit)

    @staticmethod
    def render_report_table_html(prs_info, skipped_prs_info):
        """
        :type prs_info: list[PullRequestInfoForReport]
        :type skipped_prs_info: list[PullRequestInfoForReport]
        :rtype: str
        """
        report_table_template = jinja2.Template('''
            <table border="0" style="border-collapse: collapse;">

            {% for pr_info in all_prs_info %}
                <tr>
                    <td><a href="{{ pr_info.url }}">#{{ pr_info.id }}</a>:</td>
                    <td>{{ pr_info.formatted_conflict_score_change }}</td>
                    <td>{{ ', '.join(pr_info.people) }}</td>
                </tr>
            {% endfor %}

            </table>
        ''')

        return report_table_template.render(
            all_prs_info=prs_info + skipped_prs_info,
        )

    @staticmethod
    def render_report_table_header_html(prs_info, skipped_prs_info, total_conflict_score_change):
        """
        :type prs_info: list[PullRequestInfoForReport]
        :type skipped_prs_info: list[PullRequestInfoForReport]
        :type total_conflict_score_change: int
        :rtype: str
        """
        report_table_header_template = jinja2.Template('''
            <h2>
                {{ num_pull_requests }} pull requests with conflict score {{ total_conflict_score_change }}

                {% if skipped_prs_info %}
                    (including {{ num_skipped_prs }} PRs
                    with corrupted merge-pin commits/with not affecting conflict score)
                {% endif %}
            </h2>
        ''')

        return report_table_header_template.render(
            num_pull_requests=len(prs_info) + len(skipped_prs_info),
            total_conflict_score_change=format_score(total_conflict_score_change),
            skipped_prs_info=skipped_prs_info,
            num_skipped_prs=len(skipped_prs_info),
        )

    @staticmethod
    def render_report_description_html(branch, formatted_first_day, formatted_last_day):
        """
        :type branch: str
        :type formatted_first_day: str
        :type formatted_last_day: str
        :rtype: str
        """
        report_description_template = jinja2.Template('''
            <i>Based on PRs to {{ branch }} which merged in dates {{ first_day }}&ndash;{{ last_day }}</i>
        ''')
        return report_description_template.render(
            branch=branch,
            first_day=formatted_first_day,
            last_day=formatted_last_day,
        )

    @classmethod
    def first_day(cls, num_days):
        """
        :type num_days: int
        :rtype: datetime.date
        """
        now = cls.today_midnight()
        today = now.date()
        return today - datetime.timedelta(days=num_days)

    @classmethod
    def last_day(cls):
        now = cls.today_midnight()
        today = now.date()
        return today - datetime.timedelta(days=1)

    def render_message_html(self, prs_info, skipped_prs_info, total_conflict_score_change):
        """
        :type prs_info: list[PullRequestInfoForReport]
        :type skipped_prs_info: list[PullRequestInfoForReport]
        :type total_conflict_score_change: int
        :rtype: str
        """
        report_table_header_html = self.render_report_table_header_html(
            prs_info=prs_info,
            skipped_prs_info=skipped_prs_info,
            total_conflict_score_change=total_conflict_score_change
        )
        logging.info(report_table_header_html)

        report_table_html = self.render_report_table_html(
            prs_info=prs_info,
            skipped_prs_info=skipped_prs_info
        )
        logging.info(report_table_html)

        report_description_html = self.render_report_description_html(
            branch=self.Parameters.branch,
            formatted_first_day='{:%d.%m.%Y}'.format(self.first_day(self.Parameters.days)),
            formatted_last_day='{:%d.%m.%Y}'.format(self.last_day()),
        )
        logging.info(report_description_html)

        full_html_template = jinja2.Template('''
            <html>
            <head></head>
            <body>
                {{ report_description }}
                {{ report_table_header }}
                {{ report_table }}
            </body>
            </html>
        ''')
        return full_html_template.render(
            report_description=report_description_html,
            report_table_header=report_table_header_html,
            report_table=report_table_html,
        )

    @staticmethod
    def today_midnight():
        return datetime.datetime.now().replace(
            hour=0, minute=0, second=0, microsecond=0
        )

    def render_email_message(self, message_html):
        message = MIMEText(message_html, _subtype='html', _charset='utf-8')
        message['Subject'] = '[PR statistics] Pull requests conflict scores for {}'.format(self.Parameters.branch)
        message['From'] = EMAIL_SENDER
        message['To'] = ', '.join(self.Parameters.recipients)
        return message

    def send_email(self, prs_info, skipped_prs_info, total_conflict_score_change):
        """
        :type prs_info: list[PullRequestInfoForReport]
        :type skipped_prs_info: list[PullRequestInfoForReport]
        :type total_conflict_score_change: int
        """
        message_html = self.render_message_html(
            prs_info=prs_info,
            skipped_prs_info=skipped_prs_info,
            total_conflict_score_change=total_conflict_score_change
        )
        message = self.render_email_message(message_html=message_html)

        email_client = smtplib.SMTP(SMTP_URL, SMTP_PORT)
        try:
            email_client.ehlo_or_helo_if_needed()
            email_client.sendmail(EMAIL_SENDER, self.Parameters.recipients, message.as_string())
            logging.debug('Email sent!')
        finally:
            email_client.quit()

    def get_recent_pr_merge_commits(self, target_branch, num_days):
        """
        :param target_branch: BranchFactory object
        :type num_days: int
        :return: CommitFactory object
        """
        current_commit = target_branch.latest_commit

        today_midnight_ = self.today_midnight()

        while len(current_commit.parents) > 0:
            dt = current_commit.author_timestamp
            # Enable timezone, change timezone to local, disable timezone
            dt = dt.replace(tzinfo=tz.tzutc()).astimezone(tz.tzlocal()).replace(tzinfo=None)
            if (today_midnight_ - dt).days >= num_days:
                # This PR merge commit is too old. The following PR merge commits are even older.
                logging.debug('Commit #%s is too old, stop.', current_commit.id)
                break
            if len(current_commit.parents) == 1:
                logging.debug('Commit #%s has only 1 parents, it is not a PR merge commit, skip.', current_commit.id)
                current_commit = current_commit.parents[0]
                continue
            if dt > today_midnight_:
                # This PR merge commit is too young.
                logging.debug('Commit #%s is too young, skip.', current_commit.id)
                current_commit = current_commit.parents[0]
                continue
            logging.debug('Return commit #%s', current_commit.id)
            yield current_commit
            current_commit = current_commit.parents[0]

    def get_branch_by_name(self, branch_name):
        """
        :type branch_name: str
        :return: BranchFactory object
        """
        for branch in self.bitbucket_repo.branches.get_all(filterText=branch_name):
            if branch.ref_name == 'refs/heads/{}'.format(branch_name) and branch.name == branch_name:
                logging.debug('Return branch with name %s and ref_name %s', branch.name, branch.ref_name)
                return branch
            logging.debug('Skip branch with name %s and ref_name %s', branch.name, branch.ref_name)
        assert False, 'No branch with name "{}" was found'.format(branch_name)

    def get_pr_by_pr_merge_commit(self, pr_merge_commit, target_branch_name):
        """
        :param pr_merge_commit: CommitFactory object
        :type target_branch_name: str
        :return: PullRequestFactory object or None
        :rtype: Any | None
        """
        from_branch_head_commit = pr_merge_commit.parents[1]
        logging.debug('from_branch_head_commit #%s', from_branch_head_commit.id)
        suitable_prs = []
        for pr_resource in from_branch_head_commit.pull_requests:
            pr_id = pr_resource.id
            logging.debug('pr #%s', pr_id)
            pr = self.bitbucket_repo.pull_requests[pr_id]
            if pr.state == 'MERGED' and pr.to_ref.display_id == target_branch_name:
                suitable_prs.append(pr)
        logging.debug('suitable_prs: %s', ', '.join(str(pr.id) for pr in suitable_prs))
        assert len(suitable_prs) < 2, (
            'There cannot be more than one PR associated with PR merge commit to specific branch'
        )
        if not suitable_prs:
            logging.debug('No pull requests for PR merge commit #%s', pr_merge_commit.id)
            return None
        logging.debug('Return PR #%s', pr.id)
        return suitable_prs[0]

    def on_execute(self):
        prs_info_for_report = []
        skipped_prs_info_for_report = []
        total_conflict_score_change = 0
        num_prs = 0

        target_branch = self.get_branch_by_name(self.Parameters.branch)
        # Get report data for every PR by its PR merge commit.
        for pr_merge_commit in self.get_recent_pr_merge_commits(
            target_branch=target_branch, num_days=self.Parameters.days
        ):
            pr_optional = self.get_pr_by_pr_merge_commit(
                pr_merge_commit=pr_merge_commit, target_branch_name=self.Parameters.branch
            )
            if pr_optional is None:
                logging.debug('No PR for this commit, skip.')
                continue
            pr = pr_optional

            conflict_score_change = self.get_conflict_score_change(pr=pr, pr_merge_commit=pr_merge_commit)
            pull_request_info_for_report = self.get_pull_request_info_for_report(
                pr=pr, conflict_score_change=conflict_score_change
            )

            if conflict_score_change is None:
                skipped_prs_info_for_report.append(pull_request_info_for_report)
            else:
                total_conflict_score_change += conflict_score_change
                prs_info_for_report.append(pull_request_info_for_report)

            logging.debug('PR #%d info: %s', int(pr.id), str(pull_request_info_for_report))
            num_prs += 1

        logging.info(
            (
                '%d pull requests with conflict score %d'
                ' (including %d PRs with corrupted merge-pin commits/not affecting conflict score):'
            ),
            num_prs,
            total_conflict_score_change,
            len(skipped_prs_info_for_report)
        )

        prs_info_for_report.sort(key=lambda info: -info.conflict_score_change)
        for pr_info in prs_info_for_report:
            logging.info('%s', str(pr_info))
        logging.info('Non-affecting PRs:')
        for pr_info in skipped_prs_info_for_report:
            logging.info('%s', str(pr_info))

        self.send_email(
            prs_info=prs_info_for_report,
            skipped_prs_info=skipped_prs_info_for_report,
            total_conflict_score_change=total_conflict_score_change
        )
        logging.debug('Email sent!')

        logging.info('Task BrowserSendConflictScoreReport finished')
