import json
import hashlib
import logging
import os
import shutil

import requests

import sandbox.common.errors
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt

from sandbox.projects.browser.common.git import Git
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6

from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.sdk2


REGISTRY_URL = 'registry.yandex.net'
REPO_DIR = 'repo_checkout_dir'


class AlreadyDeployedException(Exception):
    pass


class DockerImageSourceResource(sandbox.sdk2.Resource):
    """ Plaintext file resource """


class BrowserBuildAndPushToRegistry(sandbox.sdk2.Task):
    class Requirements(sandbox.sdk2.Requirements):
        disk_space = 10 * 1024
        cores = 16
        ram = 32 * 1024
        client_tags = ctc.Tag.BROWSER
        environments = [PipEnvironment('requests', '2.18.4')]

        class Caches(sandbox.sdk2.Requirements.Caches):
            pass

    class Parameters(sandbox.sdk2.Parameters):
        repo_url = sandbox.sdk2.parameters.String("Repository url", required=True)
        branch = sandbox.sdk2.parameters.String("Branch", required=True)
        targets = sandbox.sdk2.parameters.List("Targets")

        with sandbox.sdk2.parameters.Group("Credentials") as credentials_group:
            robot_login = sandbox.sdk2.parameters.String(
                "Robot to push to registry", required=True)
            robot_token_vault = sandbox.sdk2.parameters.String(
                "Vault item with robot token", required=True)

    def check_target(self, target_info):
        session = requests.Session()
        session.headers.update({'AUTHORIZATION': 'OAuth ' + sandbox.sdk2.Vault.data(
            self.Parameters.robot_token_vault)})
        tags = set(session.get('https://{}/v2/{}/tags/list'.format(
            REGISTRY_URL, target_info['image'])).json()['tags'])
        if target_info['version'] != 'latest' and target_info['version'] in tags:
            msg = 'Image {} with tag {} is already deployed'.format(
                target_info['image'], target_info['version'])
            logging.warning(msg)
            raise AlreadyDeployedException(msg)

    def create_target_resource(self, target_info):
        docker_tmp_path = 'docker_tmp_{}'.format(hashlib.md5(target_info['image']).hexdigest())
        if os.path.exists(docker_tmp_path):
            shutil.rmtree(docker_tmp_path)
        shutil.copytree(os.path.join(REPO_DIR, target_info['path']), docker_tmp_path)
        shutil.move(os.path.join(docker_tmp_path, target_info['dockerfile']),
                    os.path.join(docker_tmp_path, 'Dockerfile'))

        resource = DockerImageSourceResource(
            self, target_info['image'], docker_tmp_path
        )
        sandbox.sdk2.ResourceData(resource).ready()
        return resource

    def build_image(self, task_description):
        task_class = sandbox.sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: task_description['resource'],
            BuildDockerImageV6.RegistryTags.name: [task_description['full_name']],
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.robot_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.robot_token_vault
        }
        sub_task = task_class(
            task_class.current,
            description="Building and pushing Docker image {} (by {})".format(
                task_description['full_name'], self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        task_description['task_id'] = sub_task.id
        raise sandbox.sdk2.WaitTask(
            [sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.first_stage:
            self.Context.targets = []
            self.Context.deployed_targets = []
            Git(self.Parameters.repo_url).clone(REPO_DIR, self.Parameters.branch)

            with open(os.path.join(REPO_DIR, 'versions.json'), 'r') as f:
                config = json.load(f)
            all_targets = set(config['images'].iterkeys())
            self.Context.exit_code = 0
            if not self.Parameters.targets:
                targets = all_targets
            else:
                targets = set(self.Parameters.targets)
            unknown_targets = targets - all_targets
            if unknown_targets:
                logging.warning('%s targets are unknown; they will be ignored',
                                unknown_targets)
                targets = targets - unknown_targets
            if 'push_order' in config:
                unordered_targets = targets - set(config['push_order'])
                if unordered_targets:
                    logging.warning('%s targets are unordered; they will be ignored',
                                    unordered_targets)
                push_targets = []
                for push_target_candidate in config['push_order']:
                    if push_target_candidate not in targets:
                        continue
                    push_targets.append(push_target_candidate)
            else:
                push_targets = targets
            for push_target in push_targets:
                try:
                    push_target_info = config['images'][push_target]
                    self.check_target(push_target_info)
                    resource = self.create_target_resource(push_target_info)
                    self.Context.targets.append(
                        {'resource': resource.id,
                         'full_name': '{}/{}:{}'.format(
                             REGISTRY_URL, push_target_info['image'], push_target_info['version']),
                         'target_name': push_target,
                         'task_id': None})
                except AlreadyDeployedException as e:
                    self.set_info(e)
                    self.Context.exit_code = 1
                except Exception as e:
                    self.set_info('{} resource creation failed: {}'.format(push_target, e))
                    self.Context.exit_code = 1

        while self.Context.targets:
            current_target = self.Context.targets[0]
            if current_target['task_id'] is None:
                self.build_image(current_target)
            else:
                self.Context.targets.pop(0)
                if sandbox.sdk2.Task[current_target['task_id']].status == ctt.Status.SUCCESS:
                    self.Context.deployed_targets.append(current_target)
                else:
                    self.set_info('{} building failed'.format(current_target['target_name']))
                    self.Context.exit_code = 1

        if self.Context.deployed_targets:
            msg = '{} was/were successfully deployed'.format(
                ', '.join(target['target_name'] for target in self.Context.deployed_targets))
        else:
            msg = 'Nothing was deployed'
        self.set_info(msg)
        if self.Context.exit_code != 0:
            raise sandbox.common.errors.TaskFailure('Some targets were not deployed')
