import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc
from sandbox.common.utils import get_task_link, singleton_property
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.merge.grupper import (
    BrowserMergeScatterTests, BrowserMergeYamlToIssues
)
from sandbox.projects.browser.duty_tools import BrowserDutyUpdateBlacklists

DEFAULT_ISSUES_AUTHOR = 'robot-bro-merge'


class BrowserDutyCreateTickets(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 1 * 1024  # 1GB
        dns = misc.DnsType.DNS64

    class Parameters(RunFromYinTask.Parameters):
        tc_branch = sdk2.parameters.String(
            'Script will ask TeamCity for tests failed in builds on this branch', required=True
        )
        checkout_branch = sdk2.parameters.String(
            'Script will checkout this branch to get/commit blacklists', required=True, default='master'
        )
        num_builds = sdk2.parameters.Integer('Look through N latest builds', default=1)

        with sdk2.parameters.Group('Blacklists settings') as blacklists_settings_group:
            update_blacklists = sdk2.parameters.Bool('Update blacklists with issues', default=True)
            drop_binaries = sdk2.parameters.Bool('Do not pass whole failed binaries to the grouping tool',
                                                 default=True)
            with update_blacklists.value[True]:
                no_blacklist_commit = sdk2.parameters.Bool('Do not commit blacklist updates',
                                                           default=False)
                version = sdk2.parameters.String(
                    'Blacklist reason', description='Will group only tests with this reason'
                )
                robot_ssh_key_vault = sdk2.parameters.String('Vault item with robot token',
                                                             default='robot-bro-branch-upd_ssh_key')

        with sdk2.parameters.Group('Issue settings') as issue_settings_group:
            issues_tags = sdk2.parameters.List('Specify tags', default=[])
            replace_tags = sdk2.parameters.Bool('Replace template tags with ones from parameters',
                                                default=False)
            issues_components = sdk2.parameters.List('Specify components', default=[])
            replace_components = sdk2.parameters.Bool('Replace template components with ones from parameters',
                                                      default=False)
            title_suffix = sdk2.parameters.String('Specify title suffix')
            add_description = sdk2.parameters.String('This comment will be added at the beginning of each ticket',
                                                     default='')
            issues_priority = sdk2.parameters.String('Specify priority',
                                                     choices=[('None', None),
                                                              ('critical', 'critical'),
                                                              ('blocker', 'blocker')],
                                                     default=None)
            fix_version = sdk2.parameters.String('Specify fix version')
            ticket_template = sdk2.parameters.String(
                'Tracker ticket template',
                choices=[(t.name, t.name) for t in BrowserMergeYamlToIssues.TicketProfile],
                default=BrowserMergeYamlToIssues.TicketProfile.update_release.name,
                description='Determines how issues are pre-filled '
                            '(issue tags, components, summary, etc)')
            issues_author = sdk2.parameters.String('Issues author', default='robot-bro-branch-upd')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff('Robot login', default='robot-bro-branch-upd')
            robot_token = sdk2.parameters.String('Vault item with robot token',
                                                 default='robot-bro-branch-upd_token')

        with sdk2.parameters.Group('Debug settings') as debug_group:
            use_test_st = sdk2.parameters.Bool('Use test startrek', default=False)
            dry_run = sdk2.parameters.Bool('Dry run for BrowserMergeYamlToIssues', default=False)

        with sdk2.parameters.Output:
            grupper_addr = sdk2.parameters.String("IP address of grupper")

    class Context(RunFromYinTask.Context):
        scatter_task = None
        grupper_task = None
        yaml2issues_task = None
        update_blacklists_task = None

    @singleton_property
    def bb(self):
        return bitbucket.BitBucket(
            bitbucket.DEFAULT_BITBUCKET_URL,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.robot_token))

    @sdk2.header()
    def header(self):
        if self.Context.grupper_task:
            grupper_task = sdk2.Task.find(id=self.Context.grupper_task).first()
            return grupper_task.header()

    def on_execute(self):
        with self.memoize_stage.launch_scatter_tests():
            if (
                self.Parameters.drop_binaries or
                self.Parameters.ticket_template == BrowserMergeYamlToIssues.TicketProfile.update_release.name or
                self.Parameters.ticket_template == BrowserMergeYamlToIssues.TicketProfile.custom_tickets.name
            ):
                # we'd better not to show groups with status 'BINARY_FAILED' in branch update PRs
                drop_binaries = True
            else:
                drop_binaries = False
            scatter_task = BrowserMergeScatterTests.BrowserMergeScatterTests(
                self,
                yin_branch=self.Parameters.yin_branch,
                yin_commit=self.Parameters.yin_commit,
                tc_branch=self.Parameters.tc_branch,
                checkout_branch=self.Parameters.checkout_branch,
                launch_grupper_immediately=True,
                num_builds=self.Parameters.num_builds,
                reset_owners=False,
                ignore_blacklists=not bool(self.Parameters.version),
                version=self.Parameters.version,
                drop_ignored_tests=True,
                drop_binaries=drop_binaries,
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.robot_token,
                description='Scatter tests in {}'.format(self.Parameters.tc_branch),
                owner=self.owner,
                kill_timeout=self.Parameters.kill_timeout,
            ).enqueue()
            self.set_info(
                '<a href={}>BROWSER_MERGE_SCATTER_TESTS launched</a>'.format(get_task_link(scatter_task.id)),
                do_escape=False
            )
            self.Context.scatter_task = scatter_task.id
            raise sdk2.WaitTask(
                [scatter_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        with self.memoize_stage.wait_grupper_launched():
            scatter_task = sdk2.Task.find(id=self.Context.scatter_task).first()
            if scatter_task.status != ctt.Status.SUCCESS:
                raise TaskFailure('Scatter tests failed, cannot go any further')
            grupper_task = sdk2.Task.find(id=scatter_task.Context.grupper_task).first()
            self.set_info(
                '<a href={}>Grupper launched</a>'.format(get_task_link(grupper_task.id)),
                do_escape=False)
            self.Context.grupper_task = grupper_task.id
            targets = {self.Context.grupper_task: 'grupper_addr'}
            raise sdk2.WaitOutput(targets, wait_all=True, timeout=5 * 60)

        grupper_task = sdk2.Task.find(id=self.Context.grupper_task).first()
        if grupper_task.status == ctt.Status.TIMEOUT:
            self.set_info('Grupper timed out, finishing this task')
            return
        if grupper_task.status in ctt.Status.Group.BREAK:
            raise TaskFailure('Grupper did not start, cannot go any further')
        self.Parameters.grupper_addr = grupper_task.Parameters.grupper_addr

        with self.memoize_stage.wait_grupper_finished():
            raise sdk2.WaitTask(
                [grupper_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        if grupper_task.status == ctt.Status.TIMEOUT:
            self.set_info('Grupper timed out, finishing this task')
            return
        if grupper_task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Grupper failed, cannot go any further')

        with self.memoize_stage.create_issues():
            add_description = self.Parameters.add_description
            if self.Parameters.ticket_template == BrowserMergeYamlToIssues.TicketProfile.update_release.name:
                add_description = u'From {bb}projects/STARDUST/repos/browser/{pr}/overview\n{descr}'.format(
                    bb=bitbucket.DEFAULT_BITBUCKET_URL,
                    pr=self.Parameters.tc_branch,
                    descr=add_description
                )
            groups_resource = sdk2.Resource.find(id=grupper_task.Context.groups_resource).first()
            yaml2issues_task = BrowserMergeYamlToIssues.BrowserMergeYamlToIssues(
                self,
                yin_branch=self.Parameters.yin_branch,
                yin_commit=self.Parameters.yin_commit,
                groups_resource=groups_resource,
                verbose=True,
                update_blacklists=False,
                no_dry_run=not self.Parameters.dry_run,
                test_st_server=self.Parameters.use_test_st,
                no_wiki=True,
                replace_tags=self.Parameters.replace_tags,
                issues_tags=self.Parameters.tags,
                replace_components=self.Parameters.replace_components,
                issues_components=self.Parameters.issues_components,
                ticket_template=self.Parameters.ticket_template,
                fix_version=self.Parameters.fix_version,
                issues_author=self.Parameters.robot_login,
                title_suffix=self.Parameters.title_suffix,
                add_description=add_description,
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.robot_token,
                description="Create tickets for tests in {}".format(self.Parameters.checkout_branch),
                owner=self.owner,
            ).enqueue()
            self.set_info(
                '<a href={}>BROWSER_MERGE_YAML_TO_ISSUES launched</a>'.format(get_task_link(yaml2issues_task.id)),
                do_escape=False)
            self.Context.yaml2issues_task = yaml2issues_task.id
            raise sdk2.WaitTask(
                [yaml2issues_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        yaml2issues_task = sdk2.Task.find(id=self.Context.yaml2issues_task).first()
        if yaml2issues_task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Yaml to issues failed, cannot go any further')

        if self.Parameters.update_blacklists:
            with self.memoize_stage.update_blacklists():
                groups_with_issues_resource = sdk2.Resource.find(
                    id=yaml2issues_task.Context.groups_with_issues_resource
                ).first()
                update_blacklists_task = BrowserDutyUpdateBlacklists.BrowserDutyUpdateBlacklists(
                    self,
                    yin_branch=self.Parameters.yin_branch,
                    browser_branch=self.Parameters.checkout_branch,
                    groups_resource=groups_with_issues_resource,
                    dry_run=self.Parameters.no_blacklist_commit,
                    robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault
                ).enqueue()
                self.set_info(
                    '<a href={}>BROWSER_DUTY_UPDATE_BLACKLISTS launched</a>'.format(
                        get_task_link(update_blacklists_task.id)
                    ),
                    do_escape=False)
                self.Context.update_blacklists_task = update_blacklists_task.id
                raise sdk2.WaitTask(
                    [update_blacklists_task],
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
                )
            update_blacklists_task = sdk2.Task.find(id=self.Context.update_blacklists_task).first()
            if update_blacklists_task.status != ctt.Status.SUCCESS:
                raise TaskFailure('Updating blacklists failed, cannot go any further')
