import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc
from sandbox.common.utils import singleton_property, get_task_link
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.merge.grupper import BrowserMergeTests


class BrowserDutyUpdateBlacklists(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 1 * 1024  # 1GB
        dns = misc.DnsType.DNS64

    class Parameters(RunFromYinTask.Parameters):
        browser_branch = sdk2.parameters.String('Branch in browser repo to commit blacklists to',
                                                required=True)

        groups_resource = sdk2.parameters.Resource(
            'Sandbox resource with grouped tests', required=True,
            resource_type=BrowserMergeTests
        )
        dry_run = sdk2.parameters.Bool('Enable dry run mode (blacklists would not be changed)', default=True)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff('Robot login', default='robot-bro-branch-upd')
            robot_ssh_key_vault = sdk2.parameters.String('Vault item with robot token',
                                                         default='robot-bro-branch-upd_ssh_key')

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @singleton_property
    def git(self):
        return GitCli(self.browser_path, config={
            'user.name': self.Parameters.robot_login,
            'user.email': '{}@yandex-team.ru'.format(self.Parameters.robot_login)
        })

    def on_execute(self):
        extra_env = {
            "GIT_TRACE_PACKET": "1",
            "GIT_TRACE": "1",
            "GIT_CURL_VERBOSE": "1",
            "GIT_SSH_COMMAND": "ssh -o ServerAliveInterval=30 -o ServerAliveCountMax=4"
        }
        os.environ.update(extra_env)
        try:
            logging.info('Checkout browser branch %s', self.Parameters.browser_branch)
            vcs_root = repositories.Stardust.browser()
            vcs_root.clone(self.browser_path, self.Parameters.browser_branch)
        except subprocess.CalledProcessError:
            raise errors.TemporaryError('Checkout error')
        update_blacklists_args = [
            str(sdk2.ResourceData(self.Parameters.groups_resource).path),
            '-v',
            '--repo-root', self.browser_path
        ]
        if self.Parameters.dry_run:
            update_blacklists_args.append('--dry-run')
        logging.debug('Run yin script update_blacklists')
        self.run_yin_script('yin.sheriff.flakiness.update_blacklists',
                            script_args=update_blacklists_args, extra_env=extra_env)
        if not self.git.status('--porcelain'):
            self.set_info('No changes in blacklists were made')
            return

        try:
            if self.Parameters.dry_run:
                self.set_info('--dry-run mode, no commit will be made')
            else:
                self.git.commit(
                    '-a', '-m',
                    'Turning off broken tests. '
                    'Committed by sandbox task: {}.'.format(get_task_link(self.id)),
                )
                with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                    self.git.push('origin', '{}:refs/heads/{}'.format(self.Parameters.browser_branch,
                                                                      self.Parameters.browser_branch))
                self.set_info('Committed blacklist changes')
        except subprocess.CalledProcessError:
            raise errors.TaskFailure('Failed to commit or push changes')
