import logging

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common.keychain import MacKeychainEnvironment

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


class BrowserMacKeychainCheck(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.OSX & ~ctc.Tag.M1
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 90 * 60

        unlock = sdk2.parameters.Bool('Unlock browser keychain')
        with unlock.value[True]:
            sign = sdk2.parameters.Bool('Sign test file with browser keychain')
            check_signature = sdk2.parameters.Bool('Check signature of signed test file')

        suspend = sdk2.parameters.Bool('Suspend after keychain setup')

        with sdk2.parameters.Group('Keychain secrets'):
            browser_keychain_password = sdk2.parameters.String(
                'Browser keychain password vault',
                default=MacKeychainEnvironment.BROWSER_KEYCHAIN_PASSWORD_VAULT,
            )
            browser_keychain_content = sdk2.parameters.String(
                'Browser keychain content vault',
                default=MacKeychainEnvironment.BROWSER_KEYCHAIN_CONTENT_VAULT,
            )

    def on_execute(self):
        keychain_env = MacKeychainEnvironment(
            browser_keychain_password_vault=self.Parameters.browser_keychain_password,
            browser_keychain_content_vault=self.Parameters.browser_keychain_content,
        )
        keychain_env.prepare()

        if self.Parameters.unlock:
            keychain_env.browser_keychain.unlock()
            if self.Parameters.sign:
                test_file = self.path('test.txt')
                test_file.write_text(u'test content')
                test_filename = str(test_file)

                keychain_env.browser_keychain.sign(
                    test_filename,
                    identity='Developer ID Application: Yandex, LLC',
                )

                if self.Parameters.check_signature:
                    out = subprocess.check_output(
                        ['codesign', '--verify', '--verbose=4', test_filename],
                        stderr=subprocess.STDOUT,
                    )
                    logging.info('%s', out.decode('utf-8'))

                    out = subprocess.check_output(
                        ['codesign', '--display', '--verbose=4', test_filename],
                        stderr=subprocess.STDOUT,
                    )
                    logging.info('%s', out.decode('utf-8'))

        if self.Parameters.suspend:
            self.suspend()
