from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.browser.lib import BrowserBinTask


class BrowserPublishProject(BrowserBinTask):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BrowserBinTask.Parameters):
        arcadia_revision = sdk2.parameters.String('Arcadia revision.', required=True)
        checkout_arcadia_from_url = sdk2.parameters.String(
            'Svn url for arcadia', default='arcadia:/arc/trunk/arcadia')
        project_list = sdk2.parameters.List('Paths to projects to build.', required=True)
        resource_name = sdk2.parameters.String('Sandbox resource name.', required=True)
        output_list = sdk2.parameters.Bool('List of binaries to upload.', required=True)
        build_linux = sdk2.parameters.Bool('Build Linux binaries.', default=True)
        with build_linux.value[True]:
            output_list_linux = sdk2.parameters.List('Linux-specific list of binaries.')
            build_linux_deb = sdk2.parameters.Bool('Build debian package.', default=False)
        with build_linux_deb.value[True]:
            pkg_json = sdk2.parameters.Bool('Path to ya package json.')
            user_name = sdk2.parameters.String('User-name to sign debian package.')
        build_mac = sdk2.parameters.Bool('Build Mac binaries.', default=True)
        with build_mac.value[True]:
            output_list_mac = sdk2.parameters.List('Mac-specific list of binaries.')
        build_win = sdk2.parameters.Bool('Build Windows binaries.', default=True)
        with build_win.value[True]:
            output_list_win = sdk2.parameters.List('Win-specific list of binaries.')

    def spawn_ya_make(self, arch, output_list):
        task = sdk2.Task['YA_MAKE'](
            self,
            description='{} // Build {}'.format(
                self.Parameters.description, arch),
            owner=self.owner, priority=self.Parameters.priority,
            arcadia_revision=self.Parameters.arcadia_revision,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            arts=';'.join(output_list),
            result_rt=self.Parameters.resource_name,
            result_rd='{} // Build {} @ {}'.format(
                self.Parameters.description,
                arch, self.Parameters.arcadia_revision),
            result_single_file=bool(output_list),
            checkout_mode='auto',
            build_type='release',
            build_arch=arch,
            definition_flags=('-DNO_DEBUGINFO=yes'))
        task.enqueue()
        return task

    def spawn_ya_package(self):
        task = sdk2.Task['YA_PACKAGE'](
            self,
            description='{} // Build debian package'.format(self.Parameters.description),
            owner=self.owner, priority=self.Parameters.priority,
            arcadia_revision=self.Parameters.arcadia_revision,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            packages=self.Parameters.pkg_json,
            package_type='debian',
            publish_package=True,
            key_name=self.Parameters.user_name,
            publish_to='common',
            checkout_mode='auto',
            build_type='release')
        task.enqueue()
        return task

    def on_execute(self):
        with self.memoize_stage.run_ya_make:
            # First run. Spawn YA_MAKE tasks.
            subtasks = []
            if self.Parameters.build_linux:
                subtasks.append(self.spawn_ya_make(
                    'linux',
                    set(self.Parameters.output_list) | set(self.Parameters.output_list_linux)))
            if self.Parameters.build_mac:
                subtasks.append(self.spawn_ya_make(
                    'darwin',
                    set(self.Parameters.output_list) | set(self.Parameters.output_list_linux)))
            if self.Parameters.build_win:
                subtasks.append(self.spawn_ya_make(
                    'win',
                    set(self.Parameters.output_list) | set(self.Parameters.output_list_linux)))
            if subtasks:
                raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)
            else:
                raise sdk2.TaskFailure('No YA_MAKE tasks has been spawned!')

        with self.memoize_stage.check_ya_make_ok:
            # Second run. Fail if any YA_MAKE subtask failed.
            subtasks = list(self.find())
            if subtasks and any(task.status != ctt.Status.SUCCESS for task in subtasks):
                raise sdk2.TaskFailure('YA_MAKE task failed!')

            for task in subtasks:
                for resource in sdk2.Resource[self.Parameters.resource_name].find(task=task):
                    self.set_info(
                        'Published resource <strong>{}</strong> #{}.'.format(
                            self.Parameters.resource_name, resource.id),
                        do_escape=False)

        with self.memoize_stage.run_ya_package:
            # Second run. Spawn YA_PACKAGE task.
            if self.build_linux_deb:
                raise sdk2.WaitTask(self.spawn_ya_package(), ctt.Status.Group.FINISH, wait_all=False)

        # Last run. Fail if any subtask failed.
        subtasks = list(self.find())
        if subtasks and any(task.status != ctt.Status.SUCCESS for task in subtasks):
            raise sdk2.TaskFailure('YA_PACKAGE task failed!')
