import logging
import os

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.common.utils import singleton_property
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.localization.common import (
    prepare_localization_semaphores, BrowserLocalizationUploadedKeys,
)


class BrowserLocalizationRepoToTanker(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 4
        ram = 8 * 1024  # 8 GB
        disk_space = 32 * 1024  # 32 GB

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        package = sdk2.parameters.String('Name or path to browser-localization-core package',
                                         required=True, default='browser-localization-core==4.4.1')
        tanker_project = sdk2.parameters.String('Tanker project to use',
                                                choices=[
                                                    ('ABRO', 'abro'),
                                                    ('MOBSEARCHANDROID', 'mobsearchandroid'),
                                                    ('BROWSER', 'browser'),
                                                    ('CHROMIUM', 'browser_intl')
                                                ],
                                                required=True)

        with tanker_project.value['abro'], tanker_project.value['mobsearchandroid'], tanker_project.value['browser']:
            branch = sdk2.parameters.String('Repository branch', default='master', required=True)
        with tanker_project.value['browser_intl']:
            old_commit = sdk2.parameters.String('Old commit', required=True)
            new_commit = sdk2.parameters.String('New commit', required=True)
        recipients = sdk2.parameters.List('Recipients of status email', default=['anata-m', 'browser-infra-bots'])

        max_resource_search_depth = sdk2.parameters.Integer(
            'Look through N last published BROWSER_LOCALIZATION_UPLOADED_KEYS to find last processed commit',
            default=100)

        with sdk2.parameters.RadioGroup('Use tanker production/testing', required=True) as use_tanker_production:
            use_tanker_production.values.production = use_tanker_production.Value('Production')
            use_tanker_production.values.testing = use_tanker_production.Value('Testing', default=True)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_oauth_token = sdk2.parameters.String('Vault item with token for robot OAuth token',
                                                       default='robot-bro-locabr_oauth_token')
            robot_ssh_key_vault = sdk2.parameters.String('Vault item with ssh key for bitbucket',
                                                         default='robot-bro-locabr_ssh_key')

    def find_last_processed_commit(self, head_commit, branch, tanker_project):
        """
        We use "commit" attribute to store last successfully processed commit
        Going from HEAD down the commit history
            we try to find a resource with commit == last_processed_commit
        """
        last_processed_commit = head_commit
        for _ in range(int(self.Parameters.max_resource_search_depth)):
            logging.debug("Searching BrowserLocalizationUploadedKeys with commit attribute %s",
                          last_processed_commit)
            resource = BrowserLocalizationUploadedKeys.find(
                attrs={"commit": last_processed_commit,
                       "branch": branch,
                       "project": tanker_project}
            ).first()

            if resource:
                logging.info("Found resource with commit %s", last_processed_commit)
                return last_processed_commit
            else:
                last_processed_commit = self.git.log(
                    '-1', '--pretty=%P', last_processed_commit
                ).strip().split()[0]

        return None

    @property
    def repo_path(self):
        return str(self.path('browser'))

    @property
    def output_path(self):
        return str(self.path('tanker.xml'))

    @singleton_property
    def git(self):
        return GitCli(self.repo_path)

    def on_enqueue(self):
        prepare_localization_semaphores(self)

    def on_execute(self):
        use_tanker_production = self.Parameters.use_tanker_production == 'production'
        vcs_root = repositories.Stardust.browser()
        vcs_root.clone(self.repo_path, branch=self.Parameters.branch)

        if self.Parameters.tanker_project == 'browser_intl':
            old_commit = self.Parameters.old_commit
            new_commit = self.Parameters.new_commit
            script = 'localization_core.scripts.second_stage'
            self.git.fetch('origin')
        else:
            new_commit = self.git.rev_parse('HEAD').strip()
            old_commit = self.find_last_processed_commit(
                new_commit, self.Parameters.branch, self.Parameters.tanker_project
            ) or new_commit
            script = 'localization_core.scripts.first_stage.__main__'

        logging.info("Old commit: %s", old_commit)
        logging.info("New commit: %s", new_commit)

        env_vars = os.environ.copy()
        env_vars.update({
            'TANKER_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'BITBUCKET_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'STARTREK_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'BROWSER_BARE_REPO': self.repo_path,
        })

        with HermeticPythonEnvironment(
            python_version='2.7.17',
            packages=[self.Parameters.package],
        ) as hpe:
            with sdk2.helpers.ProcessLog(self, logger='upload_to_tanker') as pl:
                subprocess.check_call([
                    str(hpe.python_executable),
                    '-m', script,
                    old_commit, new_commit,
                    '--tanker-project', self.Parameters.tanker_project,
                    '--tanker-file', self.output_path,
                    '--use-tanker-production' if use_tanker_production else '--use-tanker-testing',
                ] + ['--recipients'] + self.Parameters.recipients, env=env_vars, stderr=pl.stdout)

        output_resource = BrowserLocalizationUploadedKeys(
            self,
            'Uploaded {} keys'.format(self.Parameters.tanker_project),
            self.output_path
        )
        if not os.path.exists(str(self.output_path)):
            # Touch a dummy file to prevent from resource break.
            with open(str(self.output_path), 'w'):
                self.set_info('WARNING: output file is empty')
        else:
            # If the task has failed and hasn't even published the resource,
            # we shouldn't use new_commit as a last successful one
            output_resource.commit = new_commit
            output_resource.branch = self.Parameters.branch
            output_resource.project = self.Parameters.tanker_project
        sdk2.ResourceData(output_resource).ready()
