import json
import os

import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import singleton_property

from sandbox.projects.browser.common.bitbucket import (
    BitBucket, DEFAULT_BITBUCKET_URL,
)
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.localization.common import REPO_TYPES
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


class BrowserLocalizationTankerToRepo(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 4
        ram = 8 * 1024  # 8 GB
        disk_space = 32 * 1024  # 32 GB

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        package = sdk2.parameters.String('Name or path to browser-localization-core package',
                                         required=True, default='browser-localization-core==4.4.1')
        tanker_branch = sdk2.parameters.String('Tanker branch', required=True, default='master')
        destination_branch = sdk2.parameters.String('Repository branch', default=None)
        keyset_type = sdk2.parameters.String('Keyset type specifies which tanker project would be used '
                                             'and which files should be updated',
                                             choices=REPO_TYPES,
                                             required=True)
        reviewers = sdk2.parameters.List('Reviewers', default=['anata-m'])
        with keyset_type.value['abro'], keyset_type.value['mobsearchandroid']:
            exclude_langs = sdk2.parameters.String(
                'Exclude languages, space-separated list',
                description='Specify languages which files should not be updated. E.g. "cs de fr"',
                default=''
            )

        with sdk2.parameters.RadioGroup('Use tanker production/testing', required=True) as use_tanker_production:
            use_tanker_production.values.production = use_tanker_production.Value('Production', default=True)
            use_tanker_production.values.testing = use_tanker_production.Value('Testing')

        with sdk2.parameters.RadioGroup('Use shuttle production/testing', required=True) as use_shuttle_production:
            use_shuttle_production.values.production = use_shuttle_production.Value('Production', default=True)
            use_shuttle_production.values.testing = use_shuttle_production.Value('Testing')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_oauth_token = sdk2.parameters.String('Vault item with token for robot OAuth token',
                                                       default='robot-bro-locabr_oauth_token')
            robot_ssh_key_vault = sdk2.parameters.String('Vault item with ssh key for bitbucket',
                                                         default='robot-bro-locabr_ssh_key')

        _container = sdk2.parameters.Container('LXC container', default_value=1414674927, required=True)

    @property
    def repo_path(self):
        return str(self.path('browser'))

    @property
    def output_path(self):
        return str(self.path('output.json'))

    @singleton_property
    def bb(self):
        return BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token', sdk2.Vault.data(self.Parameters.robot_oauth_token))

    @singleton_property
    def git(self):
        return GitCli(self.repo_path)

    def get_opened_pr(self):
        for pr in self.bb.pull_requests('stardust', 'browser', self.Parameters.destination_branch):
            if pr.title == 'Localization commit for {}'.format(self.Parameters.keyset_type):
                return pr
        return None

    def on_execute(self):
        existing_pr = self.get_opened_pr()
        if existing_pr is not None:
            self.set_info(
                '<a href={}>Pull-request</a> to {} for project {} already exists'.format(
                    existing_pr.web_url,
                    self.Parameters.destination_branch,
                    self.Parameters.keyset_type,
                ),
                do_escape=False,
            )
            return
        use_tanker_production = self.Parameters.use_tanker_production == 'production'
        use_shuttle_production = self.Parameters.use_shuttle_production == 'production'

        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            vcs_root = repositories.Stardust.browser()

            env_vars = os.environ.copy()
            env_vars.update({
                'TANKER_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
                'BITBUCKET_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
                'STARTREK_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
                'BROWSER_BARE_REPO': self.repo_path,
            })

            script_params = [
                '--tanker-branch', self.Parameters.tanker_branch,
                '--destination-branch', self.Parameters.destination_branch,
                '--use-tanker-production' if use_tanker_production else '--use-tanker-testing',
            ]
            script_params.extend(['--reviewers'] + self.Parameters.reviewers)
            if self.Parameters.exclude_langs:
                script_params.extend(['--exclude-langs'] + self.Parameters.exclude_langs.split())
            if self.Parameters.keyset_type in ['abro', 'mobsearchandroid']:
                vcs_root.clone(self.repo_path, branch=self.Parameters.destination_branch)
                script = 'localization_core.scripts.third_stage.third_stage_android'
                script_params += ['--keyset-type', self.Parameters.keyset_type]
            elif self.Parameters.keyset_type == 'repo_yandex':
                if self.Parameters.destination_branch:
                    vcs_root.clone(self.repo_path, branch=self.Parameters.destination_branch)
                else:
                    vcs_root.clone_bare(self.repo_path)
                    self.git.fetch('origin', 'refs/heads/*:refs/heads/*')
                script = 'localization_core.scripts.third_stage.third_stage_yandex'
                script_params += ['--no-compare-versions']
                script_params += ['--use-shuttle-production'] if use_shuttle_production else ['--use-shuttle-testing']
            elif self.Parameters.keyset_type == 'repo_chromium':
                vcs_root.clone_bare(self.repo_path)
                self.git.fetch('origin', 'refs/heads/*:refs/heads/*')
                script = 'localization_core.scripts.third_stage.third_stage_chromium'
                script_params += ['--repo-old-commit', self.Parameters.destination_branch,
                                  '--compare-versions']
                script_params += ['--use-shuttle-production'] if use_shuttle_production else ['--use-shuttle-testing']
            else:
                raise TaskFailure('Unknown keyset type: {}'.format(self.Parameters.keyset_type))

            with HermeticPythonEnvironment(
                    python_version='2.7.17',
                    packages=[self.Parameters.package],
            ) as hpe:
                with sdk2.helpers.ProcessLog(self, logger='download_from_tanker') as pl:
                    subprocess.check_call([str(hpe.python_executable), '-m', script] + script_params,
                                          env=env_vars, stderr=pl.stdout)

            if os.path.exists(self.output_path):
                with open(self.output_path, 'r') as output_path:
                    output_file = json.load(output_path)

                if output_file['pr_ref'] is not None:
                    self.set_info('Created <a href={}>pull-request</a>'.format(output_file['pr_ref']),
                                  do_escape=False)
                    return

            self.set_info('Pull request is unnecessary: no changes in localization files were detected')
