import os
import shutil

import sandbox.common.types.client as ctc
import sandbox.common.utils

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.browser.common.git import repositories


TANKER_TYPES = [
    ('Tanker: browser', 'tanker_yandex'),
    ('Tanker: chromium', 'tanker_chromium'),
    ('Tanker: android browser', 'abro'),
    ('Tanker: android SearchApp', 'mobsearchandroid'),
    ('Tanker: iOS browser', 'mobile-browser-ios'),
    ('Tanker: web-NTP', 'web-ntp'),
    ('Tanker: Rescue Tool', 'browser-rescue-tool'),
]

REPO_TYPES = [
    ('Repo: browser', 'repo_yandex'),
    ('Repo: chromium', 'repo_chromium'),
    ('Repo: android browser', 'abro'),
    ('Repo: android SearchApp', 'mobsearchandroid'),
]

KEYSET_TYPES = TANKER_TYPES + [
    ('Repo: browser', 'repo_yandex'),
    ('Repo: chromium', 'repo_chromium'),
    ('Repo: android browser', 'repo_abro'),
    ('Repo: android SearchApp', 'repo_mobsearchandroid'),
]


def prepare_localization_semaphores(task):
    task.Requirements.semaphores = ctt.Semaphores(
        acquires=[
            ctt.Semaphores.Acquire(name='browser_localization_upload_to_tanker')
        ],
        release=[
            ctt.Status.Group.BREAK, ctt.Status.Group.FINISH
        ]
    )


class BrowserLocalizationUploadedKeys(sdk2.Resource):
    ttl = "inf"
    commit = sdk2.Attributes.String('Last processed commit', required=True)
    branch = sdk2.Attributes.String('Branch which contains commit', required=True)
    project = sdk2.Attributes.String('Tanker project', required=True)


class BrowserLocalizationKeyset(sdk2.Resource):
    any_arch = True
    auto_backup = False
    calc_md5 = False
    executable = False
    releasable = False
    share = False
    ttl = 14


class BrowserLocalizationDisplayKeysetBase(sdk2.Task):
    JS_CONTAINER_RESOURCE = 2092551505  # TODO: somehow build this bundle

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 1
        ram = 4 * 1024  # 4 GB
        disk_space = 15 * 1024  # 15 GB

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Package') as package_group:
            package = sdk2.parameters.String('Name or path to browser-localization-core package',
                                             required=True, default='browser-localization-core==4.2.1')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_oauth_token = sdk2.parameters.String('Vault item with token for robot OAuth token',
                                                       default='robot-bro-locabr_oauth_token')

    @sandbox.common.utils.singleton_property
    def vcs_root(self):
        return repositories.Stardust.browser()

    @property
    def repo_path(self):
        return self.vcs_root.cache_repo_dir

    def keyset_dir_path(self, *args):
        return self.path('keyset', *args)

    def on_execute(self):
        os.environ.update({
            'TANKER_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'BITBUCKET_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'STARTREK_OAUTH_TOKEN': sdk2.Vault.data(self.Parameters.robot_oauth_token),
            'BROWSER_BARE_REPO': self.repo_path,
        })

    def on_prepare(self):
        self.vcs_root.update_cache_repo()
        shutil.copytree(
            str(sdk2.ResourceData(sdk2.Resource.find(id=self.JS_CONTAINER_RESOURCE).first()).path),
            str(self.keyset_dir_path())
        )
        self.keyset_dir_path().chmod(0755)
