import collections
import contextlib
import datetime
from email.mime.text import MIMEText
import itertools
import logging
import os
import smtplib
import socket

import dateutil.parser
import jinja2
import paramiko
import pytz

from sandbox import common
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status
from sandbox.common import system
from sandbox.common import utils
from sandbox.projects.browser.maintenance import client_helpers
from sandbox.projects.browser.maintenance.BrowserClientsDutyReport import filters
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


ROBOT_LOGIN = 'robot-broinfra-duty'
ROBOT_SECRET_ID = 'sec-01ezpk9pctwsb7t8hcx3n992g8'

SMTP_URL = 'yabacks.yandex.ru'
SMTP_PORT = 25
EMAIL_SUBJECT = 'Sandbox duty-bot report'
EMAIL_SENDER = ROBOT_LOGIN + '@yandex-team.ru'

ClientInRepairInfo = collections.namedtuple('ClientInRepairInfo', ('client', 'issue', 'comment'))
DeadClientInfo = collections.namedtuple('DeadClientInfo', ('client', 'problem'))
ClientWithInsufficientSpaceInfo = collections.namedtuple('ClientWithInsufficientSpaceInfo', ('client', 'free_space'))
ClientWithHungTasksInfo = collections.namedtuple('ClientWithHungTasksInfo', ('client', 'tasks'))
IdleClientInfo = collections.namedtuple('IdleClientInfo', ('client', 'idle_period'))


def ssh_execute(ssh_client, command):
    with contextlib.closing(ssh_client.get_transport().open_session()) as channel:
        channel.exec_command(command)
        with channel.makefile('r') as stdout:
            out = stdout.read()
        with channel.makefile_stderr('r') as stderr:
            err = stderr.read()
        exit_code = channel.recv_exit_status()
    return exit_code, out, err


class BrowserClientsDutyReport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 100
        environments = [
            PipEnvironment('startrek_client', version='2.5',
                           custom_parameters=['--upgrade-strategy only-if-needed']),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        client_tags = sdk2.parameters.ClientTags('Tag expression', default=Tag.BROWSER)
        free_space_threshold = sdk2.parameters.Integer('Threshold for free space on clients (in GB)')
        idle_period_threshold = sdk2.parameters.Integer('Threshold for period of doing nothing (in hours)', default=24)
        recipients = sdk2.parameters.List('Report recipients',
                                          description='Logins or mail list names, without domain name part')

        with sdk2.parameters.Group('Credentials') as credentials:
            sandbox_mac_clients_ssh_login = sdk2.parameters.String('Login for sandbox clients', required=True)
            sandbox_mac_clients_ssh_password = sdk2.parameters.Vault('Vault item with password for sandbox clients',
                                                                     required=True)
            startrek_token_secret = sdk2.parameters.YavSecret('Startrek token secret',
                                                              default=ROBOT_SECRET_ID)

    class Context(sdk2.Task.Context):
        clients = []

    def on_create(self):
        if not self.Parameters.recipients:
            self.Parameters.recipients = [self.author]

    @common.utils.singleton_property
    def startrek(self):
        import startrek_client
        return startrek_client.Startrek(
            useragent=self.__class__.name,
            token=self.Parameters.startrek_token_secret.data()['STARTREK_OAUTH_API_TOKEN'],
        )

    def get_client_data(self, chunk_size=100):
        result = []
        for offset in itertools.count(0, chunk_size):
            clients = self.server.client.read(tags=str(self.Parameters.client_tags), offset=offset, limit=chunk_size)
            if not clients['items']:
                return result
            result += clients['items']

    @utils.ttl_cache(120)
    def get_task(self, task_id):
        return self.server.task[task_id].read()

    @utils.singleton
    def find_repair_issues(self):
        """
        :return: dict with client ids as keys and repair issues as values
        :rtype: dict[str, startrek_client.objects.Resource]
        """
        mac_clients_map = {
            client['fqdn']: client
            for client in self.Context.clients
            if client_helpers.is_mac_client(client)
        }

        query = ' '.join((
            'Queue: MACFARM',
            'Status: !Closed',
            'Summary: {}'.format(', '.join(
                '#"{}"'.format(fqdn)
                for fqdn in mac_clients_map
            )),
        ))
        issues = list(self.startrek.issues.find(query=query, perScroll=100, scrollType='sorted'))

        result = {}
        for issue in issues:
            assert issue.summary in mac_clients_map
            client = mac_clients_map[issue.summary]
            result[client['id']] = issue
        return result

    def find_repair_issue(self, client):
        return self.find_repair_issues().get(client['id'])

    def is_in_repair(self, client):
        return client['id'] in self.find_repair_issues()

    def find_dead_clients(self):
        """
        Find dead clients (clients in repair are ignored).

        :return: list of dead clients
        :rtype: list[DeadClientInfo]
        """
        clients = []
        for client in self.Context.clients:
            if client['alive']:
                continue
            if self.is_in_repair(client):
                continue
            if any(client_helpers.is_maintenance_tag(tag) for tag in client['tags']):
                continue
            clients.append(DeadClientInfo(
                client=client,
                problem=self.determine_dead_client_problem(client),
            ))
        return clients

    def clients_with_maintenance_tags(self):
        """
        Find with clients with maintenance tag (clients in repair are ignored)

        :return: list of clients with maintenance tag
        :rtype: list[]
        """
        clients = []
        for client in self.Context.clients:
            if self.is_in_repair(client):
                continue
            elif any(client_helpers.is_maintenance_tag(tag) for tag in client['tags']):
                clients.append(client)
        return clients

    def determine_dead_mac_client_problem(self, client):
        ssh_password = self.Parameters.sandbox_mac_clients_ssh_password.data()

        with paramiko.SSHClient() as ssh:
            ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            try:
                ssh.connect(
                    client['fqdn'],
                    username=self.Parameters.sandbox_mac_clients_ssh_login,
                    password=ssh_password,
                    timeout=15,
                )
            except (paramiko.ssh_exception.SSHException, socket.error) as e:
                return 'SSH connection failed: {}'.format(e)

            # DB issue https://st.yandex-team.ru/TECHADMIN-2780#5dea58282fc2e411d8b1e2af
            try:
                _, output, _ = ssh_execute(ssh, 'tail -n 30 /samogon/0/active/user/agentr/srvdata/stderr | '
                                                'grep "Unable to open database file"')
            except paramiko.ssh_exception.SSHException:
                return 'Unknown problem: failed to exec SSH command'
            if output:
                return 'Unable to open agentr.db'

            # Too old date https://st.yandex-team.ru/TECHADMIN-2696#5da04788a2b79e001d17d6c2
            try:
                exit_code, output, _ = ssh_execute(ssh, 'tail -n 1 /var/log/sandbox/client.log')
            except paramiko.ssh_exception.SSHException:
                return 'Unknown problem: failed to exec SSH command'
            if exit_code != 0:
                return '/var/log/sandbox/client.log is probably missed or inaccessible (report it to sandbox)'

            try:
                # `stdout` is assumed to be like `2021-05-13 12:00:00,000 DEBUG   | ...`.
                last_client_log = datetime.datetime.strptime(' '.join(output.split()[:2]), '%Y-%m-%d %H:%M:%S,%f')
            except ValueError:
                return 'Wrong last date in /var/log/sandbox/client.log (report it to sandbox)'

            if datetime.datetime.now() - last_client_log >= datetime.timedelta(days=1):
                return 'Sandbox agent probably stopped according to /var/log/sandbox/client.log (report it to sandbox)'

        return None

    def determine_dead_client_problem(self, client):
        if client_helpers.is_mac_client(client):
            return self.determine_dead_mac_client_problem(client)
        else:
            return None

    def find_clients_with_blocking_tags(self):
        """
        Find clients with tags that block task assigning (dead clients and clients in repair are ignored).

        :return: dict with blocking tags as keys, lists of clients as values
        :rtype: dict[str, list[dict]]
        """
        result = collections.defaultdict(list)
        for client in self.Context.clients:
            if not client['alive']:
                continue
            if self.is_in_repair(client):
                continue
            for tag in filter(client_helpers.is_blocking_tag, client['tags']):
                result[tag].append(client)
        return result

    def find_clients_with_insufficient_space(self):
        """
        Find clients with insufficient free disk space.
        Note that space used by running tasks is also "free" in this context.

        :return: list of clients and free space values
        :rtype: list[ClientWithInsufficientSpaceInfo]
        """
        threshold_bytes = self.Parameters.free_space_threshold * 1024 ** 3
        clients = []

        for client in self.Context.clients:
            if not client['alive']:
                continue
            free_space = client['disk']['free_space']
            if free_space >= threshold_bytes:
                continue
            for task in client['tasks']:
                task = self.get_task(task['id'])
                # TODO: get real used space (instead of reserved) when SANDBOX-7033 will be done.
                free_space += task['requirements']['disk_space']
            if free_space < threshold_bytes:
                clients.append(ClientWithInsufficientSpaceInfo(client, free_space))

        return clients

    def find_clients_with_hung_tasks(self):
        """
        Find clients with tasks that probably hang (its kill timeout is left).

        :return: list of clients and hung tasks lists
        :rtype: list[ClientWithHungTasksInfo]
        """
        error = 60  # Error for task timeout.
        clients = []

        for client in self.Context.clients:
            hung_tasks = []
            for task in client['tasks']:
                task = self.get_task(task['id'])
                now = datetime.datetime.now(pytz.UTC)
                updated_at = dateutil.parser.parse(task['time']['updated'])
                timeout = datetime.timedelta(seconds=task['kill_timeout'] + error)
                if now - updated_at > timeout:
                    hung_tasks.append(task)
            if hung_tasks:
                clients.append(ClientWithHungTasksInfo(client, hung_tasks))

        return clients

    def find_idle_clients(self):
        """
        Find clients that do nothing for a long time.

        :return: list of clients and idle periods (in seconds)
        :rtype: list[IdleClientInfo]
        """
        threshold = self.Parameters.idle_period_threshold * 3600
        clients = []

        def get_idle_period(client):
            if client['tasks']:
                return 0
            tasks = self.server.task.read(host=client['id'], limit=1)['items']
            if not tasks:
                return float('inf')
            last_task = tasks[0]
            logging.debug('Last task on %s is #%d', client['id'], last_task['id'])
            if last_task['status'] in Status.Group.EXECUTE:
                return 0
            now = datetime.datetime.now(pytz.UTC)
            finished_time = dateutil.parser.parse(last_task['execution']['finished'])
            return (now - finished_time).total_seconds()

        for client in self.Context.clients:
            if not client['alive']:
                continue
            if any(client_helpers.is_blocking_tag(tag) for tag in client['tags']):
                continue
            if any(client_helpers.is_maintenance_tag(tag) for tag in client['tags']):
                continue
            idle_period = get_idle_period(client)
            if idle_period > threshold:
                clients.append(IdleClientInfo(client, idle_period))

        return clients

    def find_clients_in_repair(self):
        """
        Find clients with created (non-closed) repair issue.

        :return: list of clients in repair
        :rtype: list[ClientInRepairInfo]
        """
        clients = []
        for client in self.Context.clients:
            repair_issue = self.find_repair_issue(client)
            if not repair_issue:
                continue
            if client['alive']:
                comment = 'Client is already alive'
            elif not any(client_helpers.is_maintenance_tag(tag) for tag in client['tags']):
                comment = 'No maintenance tag'
            else:
                comment = ''
            clients.append(ClientInRepairInfo(
                client=client,
                issue=repair_issue,
                comment=comment,
            ))
        return clients

    def prepare_template(self):
        if system.inside_the_binary():
            from library.python import resource
            template_content = resource.find(
                'sandbox/projects/browser/maintenance/BrowserClientsDutyReport/report.html')
        else:
            template_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'report.html')
            with open(template_path, 'rb') as template_path:
                template_content = template_path.read()

        env = jinja2.Environment(extensions=['jinja2.ext.autoescape'], autoescape=True)
        env.filters.update({
            'natsort': filters.natsort,
            'parse_dt': filters.parse_dt,
            'format_dt': filters.format_dt,
            'format_dt_relative': filters.format_dt_relative,
            'format_time_delta': filters.format_time_delta,
        })
        return env.from_string(template_content)

    def send_email(self, text):
        recipients = ['{}@yandex-team.ru'.format(recipient)
                      for recipient in self.Parameters.recipients]

        message = MIMEText(text, _subtype='html', _charset='utf-8')
        message['Subject'] = EMAIL_SUBJECT
        message['From'] = EMAIL_SENDER
        message['To'] = ', '.join(recipients)

        email_client = smtplib.SMTP(SMTP_URL, SMTP_PORT)
        try:
            email_client.ehlo_or_helo_if_needed()
            email_client.sendmail(EMAIL_SENDER, recipients, message.as_string())
        finally:
            email_client.quit()

    def on_execute(self):
        self.Context.clients = self.get_client_data()

        report = self.prepare_template().render(
            tags=str(self.Parameters.client_tags),
            dead_clients=self.find_dead_clients(),
            clients_with_blocking_tags=self.find_clients_with_blocking_tags(),
            clients_with_maintenance_tags=self.clients_with_maintenance_tags(),
            clients_with_insufficient_space=self.find_clients_with_insufficient_space(),
            clients_with_hung_tasks=self.find_clients_with_hung_tasks(),
            idle_clients=self.find_idle_clients(),
            clients_in_repair=self.find_clients_in_repair(),
            current_task=self,
        )

        self.send_email(report)
