import datetime
import re

import dateutil.parser
import dateutil.tz
import jinja2


def _natsort_key(string):
    return [int(chunk) if chunk.isdigit() else chunk for chunk in re.split('([0-9]+)', string)]


@jinja2.environmentfilter
def natsort(environment, iterable, attribute=None):
    attr_getter = jinja2.filters.make_attrgetter(environment, attribute)
    return sorted(iterable, key=lambda item: _natsort_key(attr_getter(item)))


def parse_dt(dt_string):
    return dateutil.parser.parse(dt_string)


def format_dt(dt, fmt):
    return dt.strftime(fmt)


def format_dt_relative(dt):
    if dt.tzinfo is not None and dt.tzinfo.utcoffset(dt) is not None:
        now = datetime.datetime.now(dateutil.tz.tzlocal())
    else:
        now = datetime.datetime.now()
    days = (now - dt).days
    if days > 0:
        return '{} day(s) ago'.format(days)
    if days == 0:
        return 'today'
    else:
        return '{} day(s) later'.format(days)


def format_time_delta(value):
    """
    :param value: time in seconds
    :type value: float
    :return: string representation
    :rtype: str
    """
    assert value >= 0
    if value == float('inf'):
        return 'ever'
    hours = value / 3600
    days, hours = divmod(hours, 24)
    parts = []
    if days > 0:
        parts.append('{:.0f} days'.format(days))
    if hours > 0 or days == 0:
        parts.append('{:.0f} hours'.format(hours))
    return ', '.join(parts)
