import logging
import os
import platform

import yaml

from sandbox.common.config import Registry
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common import system
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status
from sandbox.common import utils

from sandbox.projects.browser import common
from sandbox.projects.browser.common import binary_tasks
from sandbox.projects.browser.common.keychain import MacKeychainEnvironment
from sandbox.projects.browser.maintenance import client_helpers
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds

from sandbox.sandboxsdk.environments import PipEnvironment, SandboxEnvironment
from sandbox import sdk2


ROBOT_SECRET_ID = 'sec-01cn4cz7j3tryd6n540b5ebgkj'  # robot-browser-infra
STARTREK_TOKEN_KEY = 'STARTREK_TOKEN'
TEAMCITY_TOKEN_KEY = 'TEAMCITY_TOKEN'

BUILD_TAG = 'pitstop'


def load_config():
    if system.inside_the_binary():
        from library.python import resource
        content = resource.find('sandbox/projects/browser/maintenance/BrowserPitstopCheck/pitstop_builds.yaml').decode()
        return yaml.safe_load(content)
    else:
        config_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'pitstop_builds.yaml')
        with open(config_path) as config_file:
            return yaml.safe_load(config_file)


class BrowserPitstopCheck(binary_tasks.CrossPlatformBinaryTaskMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.BROWSER & Tag.CUSTOM_PITSTOP
        disk_space = 100
        environments = (
            PipEnvironment('startrek_client', version='2.5',
                           custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('teamcity-client', '4.0.0'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Credentials') as credentials:
            startrek_token_secret = sdk2.parameters.YavSecret(
                'Startrek token secret',
                default=sdk2.yav.Secret(ROBOT_SECRET_ID, None, STARTREK_TOKEN_KEY))
            teamcity_token_secret = sdk2.parameters.YavSecret(
                'Teamcity token secret',
                default=sdk2.yav.Secret(ROBOT_SECRET_ID, None, TEAMCITY_TOKEN_KEY))

        _binary_task_params = binary_tasks.cross_platform_binary_task_parameters()

    class Context(sdk2.Context):
        teamcity_builds = []
        wait_task = None

    def is_first_run(self):
        return self.Context.wait_task is None

    def on_enqueue(self):
        super(BrowserPitstopCheck, self).on_enqueue()
        if not self.Requirements.host:
            # This task will be enqueued twice (second time is after WAIT_TASK) and should be executed
            # on the same host both times. So we require host to be directly specified.
            raise TaskFailure('Host should be directly specified')

    def on_prepare(self):
        if self.is_first_run():
            self.clean_caches()
        super(BrowserPitstopCheck, self).on_prepare()

    @utils.singleton_property
    def startrek(self):
        if not self.Parameters.startrek_token_secret.default_key:
            raise ValueError('Specify yav secret key for startrek token')
        import startrek_client
        return startrek_client.Startrek(
            useragent=self.__class__.name,
            token=self.Parameters.startrek_token_secret.value(),
        )

    @utils.singleton_property
    def teamcity_client(self):
        if not self.Parameters.teamcity_token_secret.default_key:
            raise ValueError('Specify yav secret key for teamcity token')
        from teamcity_client.client import TeamcityClient
        return TeamcityClient(
            server_url='https://teamcity.browser.yandex-team.ru',
            auth=self.Parameters.teamcity_token_secret.value(),
        )

    def clean_caches(self):
        for cache_dir in (
            SandboxEnvironment.build_cache_dir,
            Registry().client.vcs.dirs.base_cache,
        ):
            if not os.path.isdir(cache_dir):
                continue
            for name in os.listdir(cache_dir):
                if name == 'arc_vcs':
                    # This directory is mounted inside LXC. We cannot remove it.
                    continue
                path = os.path.join(cache_dir, name)
                logging.info('Remove %s', path)
                if os.path.isdir(path):
                    common.rmtree_forcedly(path)
                else:
                    os.remove(path)

    def check_keychain(self):
        test_file = self.path('test.txt')
        test_file.write_text(u'test content')

        keychain_env = MacKeychainEnvironment()
        keychain_env.prepare()
        keychain_env.browser_keychain.unlock()
        keychain_env.browser_keychain.sign(str(test_file), identity='Developer ID Application: Yandex, LLC')

    def check_client(self):
        if platform.system() == 'Darwin':
            self.check_keychain()

    @staticmethod
    def should_run_build(conditions):
        registry = Registry()
        return all((
            'tag' not in conditions or client_helpers.tags_match(conditions['tag'], registry.client.tags),
            conditions.get('cores', 1) <= registry.this.cpu.cores,
        ))

    @staticmethod
    def resolve_properties(properties):
        client_id = Registry().this.id
        return {
            key: value.format(client=client_id)
            for key, value in properties.iteritems()
        }

    def run_build(self, build_type, properties_map):
        build = build_type.run_build(
            comment='Triggered by {}'.format(utils.get_task_link(self.id)),
            properties=self.resolve_properties(properties_map.get(build_type.id, {})),
            snapshot_dependencies=[
                self.run_build(dep_build_type, properties_map)
                for dep_build_type in build_type.snapshot_dependencies
            ],
            tags=[BUILD_TAG],
        )
        logging.info('Triggered %s build: %s', build_type.id, build.web_url)
        return build

    def run_teamcity_builds(self):
        registry = Registry()
        logging.debug('Current client: %s', registry.this.id)
        logging.debug('Tags: %s', ', '.join(registry.client.tags))
        logging.debug('Cores number: %d', registry.this.cpu.cores)

        for build_type_data in load_config():
            if not self.should_run_build(build_type_data.get('conditions', {})):
                continue

            build_type = self.teamcity_client.BuildType(id=build_type_data['id'])

            properties_map = build_type_data.get('dep_properties', {}).copy()
            properties_map[build_type.id] = build_type_data.get('properties', {})

            build = self.run_build(build_type, properties_map)
            self.Context.teamcity_builds.append(build.id)

        if not self.Context.teamcity_builds:
            raise TaskError('No pitstop builds suitable for {}'.format(self.id))

        wait_task = BrowserWaitTeamcityBuilds(
            self,
            description='Wait pitstop builds',
            mode='WAIT_GIVEN',
            builds=' '.join(map(str, self.Context.teamcity_builds)),
        ).enqueue()
        self.Context.wait_task = wait_task.id

        raise sdk2.WaitTask(wait_task, [Status.Group.FINISH, Status.Group.BREAK])

    def check_teamcity_builds(self):
        if sdk2.Task[self.Context.wait_task].status not in Status.Group.FINISH:
            raise TaskError('Wait task unexpectedly failed')
        builds = [self.teamcity_client.Build(id=build_id) for build_id in self.Context.teamcity_builds]
        failed_builds = [build for build in builds if build.status != 'SUCCESS']
        if failed_builds:
            self.set_info('Some teamcity builds failed:\n{}'.format('\n'.join(
                '<a href="{}">{}</a>'.format(build.web_url, build.build_type.name)
                for build in failed_builds
            )), do_escape=False)
            raise TaskFailure('Some teamcity builds failed')

    def move_client_from_pitstop(self):
        self._close_startrek_issue()
        logging.info('Removing pitstop tag...')
        client_helpers.BrowserClient().remove_tags(['CUSTOM_PITSTOP'])

    def _close_startrek_issue(self):
        client_id = Registry().this.id
        client = self.server.client[client_id].read()
        issue = self._find_client_issue(client)
        if issue:
            logging.info('Closing issue %s', issue.key)
            transition = self._get_closing_transition(issue)
            transition.execute(resolution='fixed')

    def _get_closing_transition(self, issue):
        target_status_key = 'closed'
        transitions = [
            transition for transition in issue.transitions
            if transition.to.key == target_status_key
        ]
        if not transitions:
            raise ValueError('{} does not have transition to status "{}"'.format(issue.key, target_status_key))
        assert len(transitions) == 1, '{} unexpectedly has several transitions to status "{}": {}'.format(
            issue.key, target_status_key, ', '.join(transition.id for transition in transitions))
        return transitions[0]

    def _find_client_issue(self, client):
        if not client_helpers.is_mac_client(client):
            return None

        query = ' and '.join((
            'Queue: MACFARM',
            'Resolution: empty()',
            'Tags: "in_sandbox_review"',
            'Summary: #"{}"'.format(client['fqdn']),
        ))
        logging.info('Searching issues by query: %s', query)
        issues = self.startrek.issues.find(query=query, perScroll=100, scrollType='sorted')
        return next(iter(issues), None)

    def on_execute(self):
        if self.is_first_run():
            self.check_client()
            self.run_teamcity_builds()
        else:
            self.check_teamcity_builds()
            self.move_client_from_pitstop()

    def on_break(self, prev_status, status):
        super(BrowserPitstopCheck, self).on_break(prev_status, status)
        if not self.Context.teamcity_builds:
            return
        BrowserStopTeamcityBuilds(
            self,
            description='Stop pitstop builds',
            owner=self.Parameters.owner,
            teamcity_build_ids=self.Context.teamcity_builds,
            cancel_comment='Pitstop check stopped.',
        ).enqueue()
