import logging
import itertools

from sandbox.common.types.client import Tag
import sandbox.common.types.misc as ctm
from sandbox.common.types.task import Semaphores, Status
from sandbox.common import utils

from sandbox.projects.browser.maintenance import client_helpers
from sandbox.projects.browser.maintenance.BrowserPitstopCheck import BrowserPitstopCheck

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


ROBOT_SECRET_ID = 'sec-01cn4cz7j3tryd6n540b5ebgkj'  # robot-browser-infra
STARTREK_TOKEN_KEY = 'STARTREK_TOKEN'

SANDBOX_PITSTOP_TAG = Tag.CUSTOM_PITSTOP

ISSUE_PITSTOP_TAG = 'in_sandbox_pitstop'
_MAC_PITSTOP_ISSUES_BASE_QUERY = ' and '.join((
    'Queue: MACFARM',
    'Resolution: empty()',
    'Tags: "in_sandbox_review"',
    'Tags: !"{}"'.format(ISSUE_PITSTOP_TAG),
))

SANDBOX_MAC_CLIENTS_TAGS = Tag.BROWSER & Tag.Group.OSX


class BrowserPitstopTrigger(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        semaphores = Semaphores(
            acquires=(
                Semaphores.Acquire(name='BROWSER_PITSTOP_TRIGGER_SEMAPHORE', weight=1, capacity=1),
            ),
        )
        environments = [
            PipEnvironment('startrek_client', version='2.5',
                           custom_parameters=['--upgrade-strategy only-if-needed']),
        ]

    class Parameters(sdk2.Task.Parameters):

        base_pitstop_tags = sdk2.parameters.ClientTags(
            'Base pitstop tag expression', default=Tag.BROWSER,
            description='It will be combined with "{}" tag'.format(SANDBOX_PITSTOP_TAG))

        with sdk2.parameters.Group('Credentials') as credentials:
            startrek_token_secret = sdk2.parameters.YavSecret(
                'Startrek token secret',
                default=sdk2.yav.Secret(ROBOT_SECRET_ID, None, STARTREK_TOKEN_KEY))

    @utils.singleton_property
    def startrek(self):
        if not self.Parameters.startrek_token_secret.default_key:
            raise ValueError('Specify yav secret key for startrek token')
        import startrek_client
        return startrek_client.Startrek(
            useragent=self.__class__.name,
            token=self.Parameters.startrek_token_secret.value(),
        )

    def get_client_data(self, chunk_size=100, **kwargs):
        result = []
        for offset in itertools.count(0, chunk_size):
            kwargs.update(dict(
                offset=offset,
                limit=chunk_size,
            ))
            clients = self.server.client.read(**kwargs)
            if not clients['items']:
                return result
            result += clients['items']

    def move_repaired_clients_to_pitstop(self):
        for client_id, issue in self.get_pitstop_issues().items():
            logging.info('Moving %s (%s) to pitstop', client_id, issue.key)
            client_helpers.BrowserClient(client_id).add_tags([SANDBOX_PITSTOP_TAG])
            issue.update(tags={'add': [ISSUE_PITSTOP_TAG]})

    def get_pitstop_issues(self):
        """
        :return: mapping client id to startrek issue
        :rtype: dict[str, startrek_client.objects.Resource]
        """
        mac_clients = self.get_client_data(tags=str(SANDBOX_MAC_CLIENTS_TAGS))
        return self._get_mac_pitstop_issues(mac_clients)

    def _get_mac_pitstop_issues(self, clients):
        """
        :type clients: list[dict[str, Any]]
        :rtype: dict[str, startrek_client.objects.Resource]
        """
        client_id_by_fqdn = {client['fqdn']: client['id'] for client in clients}

        summaries_query = 'Summary: {}'.format(
            ', '.join(
                '#"{}"'.format(client_fqdn)
                for client_fqdn in client_id_by_fqdn
            )
        )
        query = '({}) and ({})'.format(_MAC_PITSTOP_ISSUES_BASE_QUERY, summaries_query)

        logging.info('Searching issues by query: %s', query)
        mac_issues = list(self.startrek.issues.find(query=query, perScroll=100, scrollType='sorted'))
        result = {}
        for issue in mac_issues:
            assert issue.summary in client_id_by_fqdn
            client_id = client_id_by_fqdn[issue.summary]
            result[client_id] = issue
        return result

    @utils.singleton
    def started_checks(self):
        """
        Return list of not finished BrowserPitstopCheck tasks.
        """
        query = BrowserPitstopCheck.find(
            children=True,
            status=Status.Group.ALL - Status.Group.FINISH - Status.Group.BREAK,
        )
        return query.limit(query.count)

    def is_client_busy(self, client):
        """
        :type client: dict[str, Any]
        :rtype: bool
        """
        running_task = sdk2.Task.find(
            children=True,
            host=client['id'],
            status=Status.Group.ALL - Status.Group.FINISH - Status.Group.BREAK,
        ).first()
        if running_task:
            logging.info('There is already running task on %s: %s', client['id'], running_task)
            return True

        # There can be started but not executing (ENQUEUEING, ENQUEUED or ASSIGNED) check task on target host.
        # The previous request cannot find it.
        check_task = next((task for task in self.started_checks() if task.Requirements.host == client['id']), None)
        if check_task:
            logging.info('There is already started pitstop task on %s: %s', client['id'], check_task)
            return True

        return False

    def target_clients(self):
        """
        :rtype: Iterable[str]
        """
        tags = str(self.Parameters.base_pitstop_tags & SANDBOX_PITSTOP_TAG)
        for client in self.get_client_data(alive=True, tags=tags):
            if self.is_client_busy(client):
                continue
            platform = None
            if client_helpers.is_linux_client(client):
                platform = 'linux'
            if client_helpers.is_mac_client(client):
                platform = 'mac'
            if client_helpers.is_win_client(client):
                platform = 'win'
            yield client['id'], platform

    def on_execute(self):
        self.move_repaired_clients_to_pitstop()
        skipped_clients = []
        for client_id, platform in self.target_clients():
            if platform is None:
                logging.error('Unable to determine platform of client %s', client_id)
                skipped_clients.append(client_id)
                continue
            logging.info('Triggering task on %s', client_id)
            task = BrowserPitstopCheck(self,
                                       binary_platform={
                                           'linux': ctm.OSFamily.LINUX,
                                           'mac': ctm.OSFamily.OSX,
                                           'win': ctm.OSFamily.WIN_NT,
                                       }[platform],
                                       notifications=self.Parameters.notifications)
            task.Requirements.host = client_id
            task.save()
            task.enqueue()

        if skipped_clients:
            raise RuntimeError('Unable to trigger pitstop task on following clients: {}'.format(skipped_clients))
