import logging

import sandbox.common.utils

from sandbox import sdk2


class TriggerTasksOnBrowserClients(sdk2.Task):
    """
    Run task on all alive clients with given tags (with tags from task requirements by default).
    """
    class Requirements(sdk2.Requirements):
        disk_space = 100

    class Parameters(sdk2.Task.Parameters):
        task_type = sdk2.parameters.String('Task type', required=True)
        task_params = sdk2.parameters.Dict('Task parameters')
        client_tags = sdk2.parameters.ClientTags('Client tags')

    class Context(sdk2.Task.Context):
        tasks_ids = []

    def compatible_clients(self):
        tags = (str(self.Parameters.client_tags) or
                self.server.suggest.task.read(type=self.Parameters.task_type)[0]['client_tags'])
        total = self.server.client.read(alive=True, tags=tags, limit=0)['total']
        clients = self.server.client.read(alive=True, tags=tags, limit=total)
        return (client['id'] for client in clients['items'])

    def trigger_task(self, client_id):
        logging.info('Triggering task %s on client %s', self.Parameters.task_type, client_id)
        task = self.server.task.create(
            custom_fields=[{'name': name, 'value': value} for name, value in self.Parameters.task_params.iteritems()],
            description='Triggered from <a href="{}">task #{}</a>'.format(
                sandbox.common.utils.get_task_link(self.id), self.id),
            owner=self.owner,
            requirements={'host': client_id},
            type=self.Parameters.task_type,
            notifications=self.Parameters.notifications,
        )
        logging.info('Sandbox task #%s created.', task['id'])
        return task['id']

    def on_execute(self):
        for client_id in self.compatible_clients():
            task_id = self.trigger_task(client_id)
            self.Context.tasks_ids.append(task_id)
        logging.info('Start all tasks...')
        self.server.batch.tasks.start.update(self.Context.tasks_ids)
