import contextlib
import os
import shutil

from sandbox.common import errors
from sandbox.common import share
from sandbox.common import system
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess


class UpdateBrowserWinImage(sdk2.Task):
    """
    Update Windows image for browser Teamcity agents: enable/disable agent setup on next system start
    (add or remove key to RunOnce in registry).
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.SSD
        disk_space = 70 * 1024
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 4 * 60 * 60

        image = sdk2.parameters.Resource('Image resource', resource_type=resource_types.QEMU_IMAGE)
        image_rbtorrent = sdk2.parameters.String('Image rbtorrent link')

        image_description = sdk2.parameters.String('New resource description')

        with sdk2.parameters.String('Teamcity agent setup') as agent_setup:
            agent_setup.values.enable = agent_setup.Value('enable', default=True)
            agent_setup.values.disable = agent_setup.Value('disable')

        # We need a libguestfs-tools, grub and some image. Example of preparing script:
        # apt-get update
        # apt-get install --yes grub-gfxpayload-lists libguestfs-tools
        # apt-get install --yes --reinstall linux-image-4.19.183-42.2mofed
        _container = sdk2.parameters.Container('LXC container', default=2570913136, required=False)

    def on_enqueue(self):
        if bool(self.Parameters.image) == bool(self.Parameters.image_rbtorrent):
            raise errors.TaskError('Only one of `image` and `image_rbtorrent` should be provided')

    def fetch_image(self):
        if self.Parameters.image:
            image = sdk2.ResourceData(self.Parameters.image)
            image_path = str(self.path(image.path.name))
            # Copy image because downloaded resource is read-only.
            shutil.copy(str(image.path), image_path)
            return image_path
        else:
            share.skynet_get(self.Parameters.image_rbtorrent, str(self.path()))
            name = share.files_torrent(self.Parameters.image_rbtorrent)[0]['name']
            return str(self.path(name))

    def libguestfs_env(self):
        env = os.environ.copy()
        # Make libguestfs more verbose.
        env.update({
            'LIBGUESTFS_DEBUG': '1',
            'LIBGUESTFS_TRACE': '1',
        })
        return env

    @contextlib.contextmanager
    def guestmount(self, image_path):
        mount_path = '/mnt/tmp'
        if not os.path.isdir(mount_path):
            os.makedirs(mount_path)
        with ProcessLog(self, logger='guestmount') as log:
            subprocess.check_call([
                'guestmount',
                '--format=raw',
                '--add', image_path,
                '--mount', '/dev/sda2', mount_path
            ], env=self.libguestfs_env(), stdout=log.stdout, stderr=log.stderr)
            try:
                yield mount_path
            finally:
                subprocess.check_call(['guestunmount', mount_path],
                                      env=self.libguestfs_env(), stdout=log.stdout, stderr=log.stderr)

    def get_reg_file_content(self):
        reg_file_name = {
            'enable': 'enable_agent_setup.reg',
            'disable': 'disable_agent_setup.reg',
        }[self.Parameters.agent_setup]

        if system.inside_the_binary():
            from library.python import resource
            reg_resource_path = 'sandbox/projects/browser/maintenance/UpdateBrowserWinImage/reg/{}'.format(
                reg_file_name)
            return resource.find(reg_resource_path)
        else:
            reg_file_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'reg', reg_file_name)
            with open(reg_file_path, 'rb') as reg_file:
                return reg_file.read()

    def on_execute(self):
        image_path = self.fetch_image()

        reg_file_path = self.path('agent_setup.reg')
        reg_file_path.write_bytes(self.get_reg_file_content())

        with self.guestmount(image_path) as mount_path, ProcessLog(self, logger='hivexregedit') as log:
            hive_name = 'SOFTWARE'
            prefix = '\\'.join(('HKEY_LOCAL_MACHINE', hive_name))
            hive_file_path = os.path.join(mount_path, 'Windows', 'System32', 'config', hive_name)

            subprocess.check_call([
                'hivexregedit', '--merge',
                '--prefix', prefix,
                hive_file_path,
                str(reg_file_path),
            ], env=self.libguestfs_env(), stdout=log.stdout, stderr=log.stderr)

        resource = resource_types.QEMU_IMAGE(self, self.Parameters.image_description, image_path)
        sdk2.ResourceData(resource).ready()
