import logging

from sandbox.common.config import Registry
from sandbox.common.proxy import OAuth
from sandbox.common.rest import ThreadLocalCachableClient
import sandbox.common.types.client as ctc
from sandbox.common.types.client import Tag

from sandbox import sdk2


MAINTENANCE_TAGS = frozenset(('MAINTENANCE', 'CUSTOM_MAINTENANCE'))
NEW_TAG = 'NEW'


class BrowserClient(object):
    def __init__(self, client_id=None):
        self.client_id = client_id or Registry().this.id
        self._sandbox = self._browser_infra_sandbox_client()

    @staticmethod
    def _browser_infra_sandbox_client():
        """
        Sandbox client with permission to update client tags.
        """
        sandbox_token = sdk2.Vault.data('robot-browser-infra_token')
        sandbox_auth = OAuth(sandbox_token)
        return ThreadLocalCachableClient(auth=sandbox_auth)

    def _custom_tags(self):
        tags = self._sandbox.client[self.client_id].read()['tags']
        return set(filter(is_custom_tag, tags))

    def _update_custom_tags(self, tags):
        self._sandbox.client[self.client_id].tags.update(tags)

    def update_tags(self, add_tags, remove_tags):
        """
        :type add_tags: list
        :type remove_tags: list
        """
        add_tags = set(map(str, add_tags))
        remove_tags = set(map(str, remove_tags))
        assert all(is_custom_tag(tag) for tag in add_tags | remove_tags), 'Only CUSTOM tag can be added or removed'
        tags = sorted((self._custom_tags() | add_tags) - remove_tags)
        logging.debug('Updating tags on %s: %s', self.client_id, tags)
        self._update_custom_tags(tags)

    def add_tags(self, tags):
        self.update_tags(tags, [])

    def remove_tags(self, tags):
        self.update_tags([], tags)


def tags_match(tag, client_tags):
    """
    Check if client with `client_tags` matches `tag`.

    :type tag: str
    :type client_tags: list[str]
    :rtype: bool
    """
    if hasattr(Tag, tag):
        tags = [getattr(Tag, tag)]
    elif hasattr(Tag.Group, tag):
        tags = list(getattr(Tag.Group, tag))
    else:
        raise ValueError('Unknown tag: {}'.format(tag))
    return bool(set(tags) & set(client_tags))


def is_linux_client(client):
    return bool(set(client['tags']) & set(Tag.Group.LINUX))


def is_mac_client(client):
    return bool(set(client['tags']) & set(Tag.Group.OSX))


def is_win_client(client):
    return Tag.WINDOWS in set(client['tags'])


def is_custom_tag(tag):
    return tag in ctc.Tag.Group.CUSTOM


def is_maintenance_tag(tag):
    return tag in MAINTENANCE_TAGS


def is_blocking_tag(tag):
    return (
        is_custom_tag(tag) or
        tag == NEW_TAG
    )
