import logging
import os
from os.path import isfile, join
import requests
import time
import datetime
import subprocess

import sandbox
from sandbox.projects.common import decorators
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.projects.browser.merge.BrowserApplyBackport.touch_revert_file import touch_file
from sandbox.common import system
from sandbox.projects.browser.common import LXC_RESOURCE_ID
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.common.depot_tools import ChromiumDepotToolsEnvironment
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.common.git import GitEnvironment
from sandbox.projects.browser.common.bitbucket import (
    BitBucket, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)
from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.environments import PipEnvironment

DEFAULT_CHROMIUM_PROJECT = 'CHROMIUM'

DEFAULT_CODE_MAPPING = ('a/src/', 'b/src/')
REPO_CODE_MAPPINGS = {
    'v8': ('a/src/v8/', 'b/src/v8/'),
}
MAIN_REVIEWER = 'riosvk'


class CommitInfo(object):
    def __init__(self, repo, commit_hash, code_mapping, git_cli):
        self.git_cli = git_cli
        self.code_mapping = code_mapping
        self.commit_hash = commit_hash
        self.repo = repo

    @property
    def message(self):
        return self.git_cli.show('-s', '--format=%B', self.commit_hash).strip()


class BrowserApplyBackport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 30 * 1024  # 30GB
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024
        environments = (
            GitEnvironment('2.32.0'),
            ChromiumDepotToolsEnvironment(),
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy only-if-needed']),
        )
        dns = ctm.DnsType.DNS64

        class Caches(sandbox.sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        chromium_project = sdk2.parameters.String(
            'Chromium project', default=DEFAULT_CHROMIUM_PROJECT)
        chromium_repo = sdk2.parameters.String('Chromium repo', default='src')
        chromium_commits = sdk2.parameters.List('Chromium commits', default=[])

        v8_commits = sdk2.parameters.List('V8 commits', default=[])

        main_reviewer = sdk2.parameters.String(
            'Main reviewer', default=MAIN_REVIEWER,
            description='This person will review conflicted patches and main backport pr')
        security_issue = sdk2.parameters.String(
            'Issue', required=True,
            description='Issue that initiates this backports',
        )

        with sdk2.parameters.Group('BitBucket settings') as bitbucket_group:
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_token_vault = sdk2.parameters.String(
                "Vault item with password for teamcity & bitbucket & startrek",
                default="robot-bro-merge_token")
            robot_ssh_key_vault = sdk2.parameters.String(
                "Vault item with ssh key for bitbucket",
                default="robot-bro-merge_ssh_key")

        _container = sandbox.sdk2.parameters.Container(
            'Linux container', default_value=LXC_RESOURCE_ID)

    def task_bb_href(self):
        return '[#{id}]({link})'.format(link=get_task_link(self.id), id=self.id)

    @property
    @sandbox.common.utils.singleton
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(
            bitbucket_url,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.robot_token_vault))

    @property
    @sandbox.common.utils.singleton
    def st(self):
        import startrek_client
        return startrek_client.Startrek(
            self.__class__.name, sdk2.Vault.data(self.Parameters.robot_token_vault),
            headers={'Accept-Language': 'en-US, en'}, timeout=10)

    @property
    def chromium_repo_path(self):
        return str(self.path('repo'))

    @property
    def browser_repo_path(self):
        return str(self.path('browser'))

    @property
    def v8_repo_path(self):
        return str(self.path('v8'))

    @property
    @sandbox.common.utils.singleton
    def git(self):
        return GitCli(self.browser_repo_path)

    @property
    @sandbox.common.utils.singleton
    def chromium_git(self):
        return GitCli(self.chromium_repo_path)

    @property
    @sandbox.common.utils.singleton
    def v8_git(self):
        return GitCli(self.v8_repo_path)

    def create_patches(self, commits, browser_repo_branch):
        for commit in commits:
            patch_dir = str(
                self.path('{}-{}-{}'.format(browser_repo_branch.replace('/', '-'), commit.repo, commit.commit_hash)))
            os.mkdir(patch_dir)
            commit.git_cli.fetch('origin', commit.commit_hash)

            commit.git_cli.format_patch(
                '-k', '-1',
                '--src-prefix={}'.format(commit.code_mapping[0]),
                '--dst-prefix={}'.format(commit.code_mapping[1]),
                '-o', patch_dir,
                commit.commit_hash,
            ).strip()

    def get_patch_path(self, commit, browser_repo_branch):
        patch_dir = str(
            self.path('{}-{}-{}'.format(browser_repo_branch.replace('/', '-'), commit.repo, commit.commit_hash)))
        return os.path.join(patch_dir, [f for f in os.listdir(patch_dir) if isfile(join(patch_dir, f))][0])

    def try_apply_patches(self, commits, browser_repo_branch):
        self.git.config('user.email', 'chrome-release@ya.ru')
        self.git.config('user.name', 'chromium')
        applied_commits = []
        conflicted_commits = []
        ignored_commits = []
        failed_to_apply = []
        for commit in commits:
            patch_path = self.get_patch_path(commit, browser_repo_branch)
            try:
                apply_output, err = self.git.apply('-3', '--check', patch_path)
                logging.debug('apply err: {}'.format(err))
            except subprocess.CalledProcessError as e:
                logging.error('apply err: {}'.format(e.output))
                failed_to_apply.append((commit, '{}'.format(e.output)))
            else:
                if 'with conflicts' in err:
                    conflicted_commits.append(commit)
                else:
                    self.git.apply('-3', patch_path)
                    if self.git.status('-s'):
                        applied_commits.append(commit)
                        self.git.add('-A')
                        self.git.commit('-m', commit.message)
                    else:
                        ignored_commits.append(commit)
        return applied_commits, conflicted_commits, ignored_commits, failed_to_apply

    def create_prs_for_conflicted_commits(self, commits, main_backports_branch, browser_repo_branch):
        prs = []
        for commit in commits:
            patch_path = self.get_patch_path(commit, browser_repo_branch)
            branch = 'wp/robots/backport_{}/{}'.format(commit.commit_hash[:8], int(time.time() * 1000))
            self.git.checkout('-b', branch, main_backports_branch)
            try:
                self.git.apply('-3', patch_path)
            except Exception:
                # conflicts
                pass
            self.git.add('-A')
            self.git.commit('-m', commit.message)
            self.push(branch)
            title = '{}: backport {}/{}'.format(
                self.Parameters.security_issue, commit.repo, commit.commit_hash)
            pr = self.bb.create_pr('STARDUST', 'browser', title,
                                   '', branch, main_backports_branch, [self.Parameters.main_reviewer])

            # 35 is id of the rule that turns almost all vetos off
            self.bb.skip_vetoes_in_pr('STARDUST', 'browser', pr.id, 35)
            prs.append(pr)
        return prs

    def touch_force_revert_file(self):
        force_revert_path = os.path.join(self.browser_repo_path, 'src', 'force_branch_update_revert.yaml')
        if system.inside_the_binary():
            touch_file(force_revert_path, self.Parameters.security_issue)
        else:
            with HermeticPythonEnvironment(
                python_version='2.7.17',
                pip_version='9.0.1',
                packages=('ruamel.ordereddict==0.4.13', 'ruamel.yaml==0.15.35'),
            ) as hpe:
                script_path = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'touch_revert_file.py')
                command = [str(hpe.python_executable), script_path] + [
                    force_revert_path,
                    self.Parameters.security_issue,
                ]
                logging.debug('Running {}'.format(command))
                env = os.environ.copy()
                with sdk2.helpers.ProcessLog(self, logger="script-log") as pl:
                    subprocess.check_call(command, env=env, stdout=pl.stdout, stderr=pl.stdout)
        self.git.config('user.email', 'robot-bro-merge@yandex-team.ru')
        self.git.config('user.name', 'robot-bro-merge')
        self.git.add('src/force_branch_update_revert.yaml')
        self.git.commit('-m', 'force revert')

    @decorators.retries(5, delay=1, backoff=2)
    def set_fixversion(self, version):
        issue_obj = self.st.issues[self.Parameters.security_issue]
        issue_fix_versions = {version.name for version in issue_obj.fixVersions}
        if version not in issue_fix_versions:
            issue_obj.update(
                params={'notify': False},
                fixVersions={
                    'add': [version]
                }
            )

    def get_releases(self):
        releases_url = (
            'https://browser.yandex-team.ru/rest/releases/'
            '?show_only_active=true&include_milestones=true'
        )
        current_timestamp = int(time.mktime(datetime.datetime.today().timetuple()))
        current_releases = requests.get(releases_url).json()['items']
        return [
            release
            for release in current_releases
            if (
                any(
                    milestone['kind'] == 'branch' and milestone['date'] < current_timestamp
                    for milestone in release['milestones']
                )
            )
        ]

    def push(self, branch):
        try:
            with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                self.git.push('origin', branch)
        except Exception:
            logging.exception('Failed to push commit to %s', branch)
            raise TaskFailure('Failed to push commit to {}'.format(branch))

    def clone_repos(self):
        vcs_root = repositories.bitbucket_vcs_root(
            self.Parameters.chromium_project,
            self.Parameters.chromium_repo,
            testing=self.Parameters.use_test_bitbucket)
        vcs_root.clone(self.chromium_repo_path, 'main')

        vcs_root = Git('https://chromium.googlesource.com/v8/v8')
        vcs_root.clone(self.v8_repo_path, 'main')

        browser_vcs_root = repositories.Stardust.browser(testing=self.Parameters.use_test_bitbucket)
        browser_vcs_root.clone(self.browser_repo_path, 'master')

    def get_commits_info(self):
        commits = [
            CommitInfo(self.Parameters.chromium_repo, commit,
                       REPO_CODE_MAPPINGS.get(self.Parameters.chromium_repo, DEFAULT_CODE_MAPPING),
                       self.chromium_git)
            for commit in self.Parameters.chromium_commits
        ]
        commits += [
            CommitInfo('v8', commit,
                       REPO_CODE_MAPPINGS.get('v8', DEFAULT_CODE_MAPPING),
                       self.v8_git)
            for commit in self.Parameters.v8_commits
        ]
        return commits

    def create_main_pr(self, conflicted_prs, main_branch, good_commits, ignored_commits, failed_to_apply,
                       target_branch, main_master_pr):
        self.git.checkout(main_branch)
        self.git.commit('-m', 'empty commit', '--allow-empty')  # to be sure pr will be created
        self.push(main_branch)
        if target_branch == 'master':
            self.touch_force_revert_file()
            self.push(main_branch)
        reviewers = [self.Parameters.main_reviewer]
        description = 'sandbox task: {}\n'.format(self.task_bb_href())
        if main_master_pr:
            description += 'Pr to master: {}\n'.format(main_master_pr.web_url)
        if good_commits:
            description += 'Automatically applied commits:\n'
            description += '\n'.join('{}/{}'.format(commit.repo, commit.commit_hash) for commit in good_commits)
            description += '\n'
        if ignored_commits:
            description += 'Commits that produced zero diff (seems that we already have these changes):\n'
            description += '\n'.join('{}/{}'.format(commit.repo, commit.commit_hash) for commit in ignored_commits)
            description += '\n'
        if failed_to_apply:
            description += 'Failed to apply commits (`git apply --check` returned non-zero exit status):\n'
            description += '\n'.join(
                '{}/{} Error: {}'.format(commit.repo, commit.commit_hash, error) for commit, error in failed_to_apply)
            description += '\n'
        if conflicted_prs:
            description += 'Some commits conflicted. Created separate prs for them:\n'
            description += '\n'.join(pr.links['self'][0]['href'] for pr in conflicted_prs)
            description += '\n\nDo not merge this pr before other prs!'
        return self.bb.create_pr('STARDUST', 'browser', 'Apply backports', description, main_branch, target_branch,
                                 reviewers)

    def apply_all_patches_and_create_pr(self, chromium_commits, browser_repo_branch, main_master_pr):
        self.create_patches(chromium_commits, browser_repo_branch)
        main_backport_branch = 'wp/robots/{}/{}/{}'.format(self.Parameters.security_issue, browser_repo_branch,
                                                           int(time.time() * 1000))
        self.git.checkout('-b', main_backport_branch)
        applied_commits, conflicted_commits, ignored_commits, failed_to_apply = (
            self.try_apply_patches(chromium_commits, browser_repo_branch))
        main_pr = None
        if conflicted_commits or applied_commits:
            self.push(main_backport_branch)
            prs = self.create_prs_for_conflicted_commits(conflicted_commits, main_backport_branch, browser_repo_branch)
            main_pr = self.create_main_pr(prs, main_backport_branch,
                                          applied_commits, ignored_commits, failed_to_apply,
                                          browser_repo_branch, main_master_pr)
        else:
            self.set_info('All commits seem to be already in {}. Will create no prs.'.format(browser_repo_branch))
        return main_pr

    def on_execute(self):
        self.clone_repos()
        commits = self.get_commits_info()
        main_master_pr = self.apply_all_patches_and_create_pr(commits, 'master', None)
        for release in self.get_releases():
            branch = release['branch']
            self.git.fetch('origin', branch)
            self.git.checkout('origin/' + branch)
            main_pr = self.apply_all_patches_and_create_pr(commits, branch, main_master_pr)
            if main_pr:
                self.set_fixversion(release['version'])
        if main_master_pr and not self.get_releases():
            self.set_fixversion('Master')
