from collections import defaultdict
import copy
import logging

import sandbox.common.types.client as ctc

from sandbox.projects.browser.common.bitbucket import (
    BitBucket, ResourceNotFound, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)
from sandbox.projects.browser.common.chromium_releases import (
    ChromiumDash, Release, DEFAULT_TRACKED_CHANNELS, DEFAULT_TRACKED_PLATFORMS)
from sandbox.projects.browser.merge.BrowserMakeChromiumSnapshot import (
    BrowserMakeChromiumSnapshot)

from sandbox import sdk2

DEFAULT_SNAPSHOTS_PROJECT = 'STARDUST'
DEFAULT_SNAPSHOTS_REPO = 'chromium-snapshots'
DEFAULT_CHROMIUM_PROJECT = 'CHROMIUM'
DEFAULT_CHROMIUM_REPO = 'src'


class BrowserFetchChromiumVersions(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100  # 100MB. There are no real disk usage in this task.
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        restrict_branches = sdk2.parameters.List(
            'Restrict branches', default=[])
        with sdk2.parameters.Group('ChromiumDash settings') as chromiumdash_group:
            chromiumdash_url = sdk2.parameters.String(
                'ChromiumDash url', default=ChromiumDash.releases_url)
            chromiumdash_tracked_platforms = sdk2.parameters.List(
                'Tracked platforms', default=DEFAULT_TRACKED_PLATFORMS)
            chromiumdash_tracked_channels = sdk2.parameters.List(
                'Tracked channels', default=DEFAULT_TRACKED_CHANNELS)

        with sdk2.parameters.Group('BitBucket settings') as bitbucket_group:
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
            snapshots_project = sdk2.parameters.String(
                'Project', default=DEFAULT_SNAPSHOTS_PROJECT)
            snapshots_repo = sdk2.parameters.String(
                'Repository', default=DEFAULT_SNAPSHOTS_REPO)
            dt_repo = sdk2.parameters.String(
                'depot_tools repository', default='depot_tools')
            dt_reviewers = sdk2.parameters.List('Depot tools Reviewers.')
            chromium_project = sdk2.parameters.String(
                'Chromium project', default=DEFAULT_CHROMIUM_PROJECT)
            chromium_repo = sdk2.parameters.String(
                'Chromium repo', default=DEFAULT_CHROMIUM_REPO)
            chromium_dt_repo = sdk2.parameters.String(
                'Chromium depot_tools repo', default='tools-depot_tools')

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_login = sdk2.parameters.String(
                "Login for teamcity & bitbucket", default="robot-bro-merge")
            robot_password_vault = sdk2.parameters.String(
                "Vault item with password for teamcity & bitbucket",
                default="robot-bro-merge_password")
            robot_ssh_key_vault = sdk2.parameters.String(
                "Vault item with ssh key for bitbucket",
                default="robot-bro-merge_ssh_key")

    @property
    def bb(self):
        if getattr(self, '_bb', None) is None:
            if self.Parameters.use_test_bitbucket:
                bitbucket_url = TESTING_BITBUCKET_URL
            else:
                bitbucket_url = DEFAULT_BITBUCKET_URL
            self._bb = BitBucket(
                bitbucket_url,
                self.Parameters.robot_login,
                sdk2.Vault.data(self.Parameters.robot_password_vault))
        return self._bb

    def get_current_release_for_branch(self, branch):
        '''
        Fetches src/chrome/VERSION from branch of snapshots repository.
        Returns parsed object.

        Return None if there are no branch or file.
        '''

        try:
            version_file = self.bb.load_file(
                self.Parameters.snapshots_project,
                self.Parameters.snapshots_repo,
                'src/chrome/VERSION',
                at='refs/heads/' + branch)
            release = Release.fromversionfile(version_file, set())
            self.set_info(
                'Current version in <strong>{}</strong> is '
                '<strong>{}</strong>'.format(
                    branch, release.version),
                do_escape=False)
            return release
        except ResourceNotFound:
            return None

    def get_snapshots_branches(self):
        '''
        Returns dict of branch/commit in snapshots repository.
        '''

        return {
            branch['displayId']: branch['latestChangeset']
            for branch in self.bb.get_branches(
                self.Parameters.snapshots_project,
                self.Parameters.snapshots_repo)
        }

    def on_execute(self):
        releases = self.fetch_actual_releases()
        current_releases = self.fetch_current_releases(releases)
        chains_starts, chains = self.make_chains(releases, current_releases)
        chains = self.filter_branches(chains)

        tasks = self.spawn_tasks(chains, chains_starts)

        for version, task in sorted(tasks.items()):
            self.set_info(
                'Release <strong>{}</strong> will be imported '
                'in task <strong>#{}</strong>.'.format(
                    version, task.id),
                do_escape=False)

    def filter_branches(self, chains):
        if not self.Parameters.restrict_branches:
            return chains

        chains = {
            branch: chains[branch]
            for branch in self.Parameters.restrict_branches
            if branch in chains
        }

        # If we filtered out all chains except upstream/dev, we must copy
        # it to other branch, because spawn_tasks will not spawn tasks
        # for upstream/dev itself.
        if chains.keys() == ['upstream/dev'] and chains['upstream/dev']:
            branch = chains['upstream/dev'][-1].upstream_branch
            chains[branch] = chains['upstream/dev']

        return chains

    def fetch_actual_releases(self):
        '''
        Fetches, aggregates and returns list of releases from
        ChromiumDash (dev, beta, stable, extended channels).
        '''

        chromiumdash = ChromiumDash(
            self.Parameters.chromiumdash_tracked_platforms,
            self.Parameters.chromiumdash_tracked_channels
        )

        return chromiumdash.releases_for_channel(chromiumdash.tracked_channels)

    def fetch_current_releases(self, releases):
        '''
        Fetches from bitbucket last imported releases.
        '''

        snapshots_branches = self.get_snapshots_branches()
        current_releases = {
            'upstream/dev': self.get_current_release_for_branch('upstream/dev')
        }
        for release in releases.values():
            branch = release.upstream_branch
            if branch in current_releases:
                continue

            if branch not in snapshots_branches:
                continue

            current_releases[branch] = self.get_current_release_for_branch(
                branch)

        return current_releases

    def make_chains(self, releases, current_releases):
        '''
        Groups releases by major version and makes chains of import.
        '''

        chains = defaultdict(list)
        start = copy.copy(current_releases)
        last_releases = copy.copy(start)

        for release in sorted(releases.values()):
            logging.info('Processing %s', release.version)
            branch = release.upstream_branch
            if (branch in current_releases and
                    current_releases[branch] >= release):
                self.set_info(
                    'Current release in <strong>{}</strong> '
                    '(<strong>{}</strong>) is new enough. Skip '
                    'import of <strong>{}</strong>.'.format(
                        branch,
                        current_releases[branch].version,
                        release.version),
                    do_escape=False)
                continue

            assert (branch not in last_releases or
                    last_releases[branch] < release)

            if branch not in start:
                if last_releases['upstream/dev'] >= release:
                    self.set_info(
                        'Current release in <strong>upstream/dev</strong> '
                        '(<strong>{}</strong>) is greater than needed. '
                        'skip release <strong>{}</strong>.'.format(
                            last_releases['upstream/dev'].version,
                            release.version),
                        do_escape=False)
                    continue

                start[branch] = last_releases['upstream/dev']

            if branch not in last_releases:
                last_releases[branch] = start[branch]

            if ('dev' in release.channels and
                    last_releases['upstream/dev'] == last_releases[branch]):
                last_releases['upstream/dev'] = release
                chains['upstream/dev'].append(release)

            last_releases[branch] = release
            chains[branch].append(release)

        return start, chains

    def spawn_tasks(self, chains, start):
        '''
        Creates and starts tasks for snapshot importing.
        '''

        tasks = {}
        for branch, chain in sorted(chains.items()):
            if branch == 'upstream/dev':
                continue

            prev_release = start[branch]
            for release in chain:
                prev_task = None
                if prev_release is not None:
                    prev_task = tasks.get(prev_release.version)

                description = 'Make snapshot of chromium version {}'.format(
                    release.version)
                if prev_task:
                    description += ' depends on #{}'.format(prev_task.id)

                target_branches = [branch]
                if release in chains['upstream/dev']:
                    target_branches.append('upstream/dev')

                task = BrowserMakeChromiumSnapshot(
                    self,
                    owner=self.Parameters.owner,
                    description=description,
                    notifications=self.Parameters.notifications,
                    previous_task=prev_task.id if prev_task else None,
                    previous_version=(
                        prev_release.version if prev_release else None),
                    target_branches=target_branches,
                    version=release.version,
                    robot_login=self.Parameters.robot_login,
                    robot_password_vault=self.Parameters.robot_password_vault,
                    robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault,
                    use_test_bitbucket=self.Parameters.use_test_bitbucket,
                    snapshots_project=self.Parameters.snapshots_project,
                    snapshots_repo=self.Parameters.snapshots_repo,
                    dt_repo=self.Parameters.dt_repo,
                    dt_reviewers=self.Parameters.dt_reviewers,
                    chromium_project=self.Parameters.chromium_project,
                    chromium_repo=self.Parameters.chromium_repo,
                    chromium_dt_repo=self.Parameters.chromium_dt_repo)
                tasks[release.version] = task
                prev_release = release

        for task in tasks.values():
            task.enqueue()

        return tasks
