import collections
import functools
import logging
import os
import shutil
import subprocess

from sandbox import common

import sandbox.common.errors
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.sdk2

from sandbox.projects.browser.common import LXC_RESOURCE_ID
from sandbox.projects.browser.common.bitbucket import (
    BitBucket, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)
from sandbox.projects.browser.common.chromium_releases import Release
from sandbox.projects.browser.common.depot_tools import ChromiumDepotToolsEnvironment
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli

from sandbox.projects.browser.merge.BrowserMergeDepotTools import BrowserMergeDepotTools
from sandbox.projects.browser.merge.common import python_seriallization
from sandbox.projects.browser.merge.common import deps


DEFAULT_SNAPSHOTS_PROJECT = 'STARDUST'
DEFAULT_SNAPSHOTS_REPO = 'browser'
DEFAULT_DEPS_PROJECT = 'STARDUST'

SNAPSHOT_REVISION_FILE = '.lastchange'


def action(message):
    '''
    Decorator that simply applies sdk2.helpers.misc.ProgressMeter context manager.
    '''

    def decorator(f):
        @functools.wraps(f)
        def wrapper(*args, **kwargs):
            with sandbox.sdk2.helpers.ProgressMeter(message):
                return f(*args, **kwargs)
        return wrapper
    return decorator


class BrowserMakeChromiumSnapshot(sandbox.sdk2.Task):
    class Requirements(sandbox.sdk2.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024
        environments = (
            GitEnvironment('2.32.0'),
            ChromiumDepotToolsEnvironment(),
        )
        dns = ctm.DnsType.DNS64

        class Caches(sandbox.sdk2.Requirements.Caches):
            pass

    class Parameters(sandbox.sdk2.Task.Parameters):
        kill_timeout = 5 * 3600
        target_branches = sandbox.sdk2.parameters.List(
            'Branch to commit snapshot to.')

        version = sandbox.sdk2.parameters.String(
            'Chromium version to import.', required=True)
        previous_task = sandbox.sdk2.parameters.Integer(
            'Wait task with given id before start.',
            required=False, default=None)
        previous_version = sandbox.sdk2.parameters.String(
            'Previous snapshot version.', required=False)
        gclient_sync_jobs = sandbox.sdk2.parameters.Integer(
            'Gclient sync jobs', default=4)
        skip_dt_merge = sandbox.sdk2.parameters.Bool(
            'Skip depot_tools merge', default=False)
        force = sandbox.sdk2.parameters.Bool(
            'Make snapshot without checking current snapshot version.',
            default=False)

        with sandbox.sdk2.parameters.Group('BitBucket settings'):
            use_test_bitbucket = sandbox.sdk2.parameters.Bool(
                'Use test BitBucket')
            snapshots_project = sandbox.sdk2.parameters.String(
                'Project', default=DEFAULT_SNAPSHOTS_PROJECT)
            snapshots_repo = sandbox.sdk2.parameters.String(
                'Repository', default=DEFAULT_SNAPSHOTS_REPO)
            dt_repo = sandbox.sdk2.parameters.String(
                'depot_tools repository', default='depot_tools')
            dt_reviewers = sandbox.sdk2.parameters.List(
                'Depot tools Reviewers.')
            chromium_project = sandbox.sdk2.parameters.String(
                'Chromium project', default='CHROMIUM')
            chromium_repo = sandbox.sdk2.parameters.String(
                'Chromium repo', default='src')
            chromium_dt_repo = sandbox.sdk2.parameters.String(
                'Chromium depot_tools repo', default='tools-depot_tools')

        with sandbox.sdk2.parameters.Group("Credentials"):
            robot_login = sandbox.sdk2.parameters.String(
                "Login for teamcity & bitbucket", default="robot-bro-merge")
            robot_password_vault = sandbox.sdk2.parameters.String(
                "Vault item with password for teamcity & bitbucket",
                default="robot-bro-merge_password")
            robot_ssh_key_vault = sandbox.sdk2.parameters.String(
                "Vault item with ssh key for bitbucket",
                default="robot-bro-merge_ssh_key")

        with sandbox.sdk2.parameters.Group("Debug"):
            suspend_after_snapshot = sandbox.sdk2.parameters.Bool(
                'Suspend after creating snapshot', default=False)
            enable_breakpoints = sandbox.sdk2.parameters.Bool(
                'Debug mode: suspend on each breakpoint defined in code', default=False)

        _container = sandbox.sdk2.parameters.Container(
            'Linux container', default_value=LXC_RESOURCE_ID)

    def on_enqueue(self):
        '''
        Wait untill task with given id is finished.
        '''

        previous_task_id = self.Parameters.previous_task
        if previous_task_id:
            self.wait_task_by_id(previous_task_id)

        merge_depot_tools_task_id = self.Context.merge_depot_tools_task
        if merge_depot_tools_task_id:
            self.wait_task_by_id(merge_depot_tools_task_id)

    def wait_task_by_id(self, task_id):
        logging.info('Looking for task #%d', task_id)
        task = sandbox.sdk2.Task.find(
            id=task_id, children=True).limit(1).first()

        if not task:
            raise sandbox.common.errors.TaskFailure(
                'Task {} not found!'.format(task_id))

        if task.status not in ctt.Status.Group.FINISH:
            logging.info(
                'Task #%d is in state %s. Waiting for finish.',
                task.id, task.status)
            raise sandbox.sdk2.WaitTask(task, ctt.Status.Group.FINISH, True)

    def fetch_remote_branches(self):
        '''
        Return dict of branch/commit from origin.
        '''

        ls_remote_output = self.git.ls_remote('--heads', 'origin', 'upstream/*')

        branches = {}
        for line in ls_remote_output.split('\n'):
            if not line:
                continue

            commit, ref = line.split('\t', 1)
            commit = commit.strip()
            ref = ref.strip()
            branches[ref[len('refs/heads/'):]] = commit

        return branches

    @property
    def repo_path(self):
        return str(self.path('chromium-snapshots'))

    @property
    def gclient_cache(self):
        return str(self.path('gclient_cache'))

    @property
    @common.utils.singleton
    def git(self):
        return GitCli(self.repo_path, config={
            'user.name': 'chromium',
            'user.email': 'chrome-release@ya.ru'
        })

    def gclient(self, *args):
        '''
        Helper for running gclient in the repository.
        '''

        with sandbox.sdk2.helpers.ProcessLog(
                self, logger=logging.getLogger('gclient')) as pl:
            pl.logger.propagate = 1
            return subprocess.check_call(
                ['gclient'] + list(args),
                stdout=pl.stdout, stderr=subprocess.STDOUT,
                cwd=self.repo_path)

    @property
    @common.utils.singleton
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(
            bitbucket_url,
            self.Parameters.robot_login,
            sandbox.sdk2.Vault.data(
                self.Parameters.robot_password_vault))

    def debug_breakpoint(self, label=None):
        if self.Parameters.enable_breakpoints:
            if label:
                logging.info('Debug breakpoint: %s', label)
            self.suspend()

    def make_snapshot(self):
        if self.Parameters.force:
            logging.warning('FORCE SNAPSHOT')
            self.set_info(
                '<strong style="color: red;">FORCE SNAPSHOT</strong>',
                do_escape=False)
        logging.info('Starting import...')
        for branch in self.Parameters.target_branches:
            logging.info('%s -> %s', self.Parameters.version, branch)
            self.set_info(
                '<strong>{version}</strong> -&gt; '
                '<strong>{branch}</strong>'.format(
                    version=self.Parameters.version, branch=branch),
                do_escape=False)

        current_commit = self.clone_repo()
        for branch in self.Parameters.target_branches:
            if (branch in self.remote_branches and
                    self.remote_branches[branch] != current_commit):
                logging.error(
                    'Desired branch (%s) points to different commit: '
                    '%s, %s is expected',
                    branch, self.remote_branches[branch], current_commit)
                raise sandbox.common.errors.TaskFailure(
                    'Desired branch ({}) points to different commit: '
                    '{}, {} is expected'.format(
                        branch, self.remote_branches[branch], current_commit))

        current_release = self.get_current_release()
        if current_release > self.target_release:
            logging.error(
                'Can not snapshot %s over %s',
                self.target_release.version, current_release.version)
            raise sandbox.common.errors.TaskFailure(
                'Can not snapshot {} over {}'.format(
                    current_release.version, self.target_release.version))
        elif (current_release == self.target_release and
                not self.Parameters.force):
            logging.error(
                'Can not snapshot %s over %s',
                self.target_release.version, current_release.version)
            raise sandbox.common.errors.TaskFailure(
                'Can not snapshot {} over {}'.format(
                    current_release.version, self.target_release.version))
        elif (current_release == self.target_release and self.Parameters.force):
            logging.warning(
                'Forcing snapshot creation! '
                'Current release: %s, target release: %s',
                current_release.version, self.target_release.version)
            self.set_info(
                '<strong span="color: red;">Forcing snapshot {version} '
                'over {current_version}</strong>'.format(
                    version=self.target_release.version,
                    current_version=current_release.version),
                do_escape=False)

        if self.Parameters.previous_version:
            previous_release = Release(
                self.Parameters.previous_version, None, set())
            if previous_release != current_release:
                logging.error(
                    'Current release mismatch. %s is expected when %s found.',
                    previous_release.version, current_release.version)
                raise sandbox.common.errors.TaskFailure(
                    'Current release mismatch. {} is expected when {} '
                    'found.'.format(
                        previous_release.version, current_release.version))

        old_deps = self.load_deps_snapshot()
        if 'src/third_party/depot_tools' in old_deps.deps:
            depot_tools_dep = old_deps.deps['src/third_party/depot_tools']
            if isinstance(depot_tools_dep, dict):
                old_depot_tools_revision = \
                    depot_tools_dep['url'].rsplit('@', 1)[-1]
            else:
                old_depot_tools_revision = depot_tools_dep.rsplit('@', 1)[-1]
            if old_depot_tools_revision.startswith('remotes/origin/'):
                old_depot_tools_revision = (
                    old_depot_tools_revision[len('remotes/origin/'):])
            if old_depot_tools_revision.startswith('upstream-merge/'):
                old_depot_tools_revision = (
                    old_depot_tools_revision[len('upstream-merge/'):])
        else:
            old_depot_tools_revision = None

        self.cleanup_repository()
        self.checkout_src()
        with open(os.path.join(self.repo_path, 'src', 'DEPS')) as f:
            chromium_deps = deps.Deps.from_string(f.read())
        flatten_input_deps = chromium_deps.filter_out_cipd()
        flatten_input_deps.save(os.path.join(self.repo_path, 'src', 'DEPS.patched'))
        self.configure_gclient_for_chromium('DEPS.patched')
        self.sync_chromium()
        flatten_output_deps = self.flatten_chromium_deps()
        chromium_deps.deps.update(flatten_output_deps.deps)
        chromium_deps.vars.update(flatten_output_deps.vars)
        chromium_deps = chromium_deps.filter_out_recursedeps()  # no need in recursedeps after flatten
        with open(os.path.join(self.repo_path, 'src', 'third_party', 'skia', 'DEPS')) as f:
            skia_deps = deps.Deps.from_string(f.read())
        for path, dep in skia_deps.deps.iteritems():
            if path == '../src':
                continue
            chromium_deps.deps['src/third_party/skia/' + path] = dep
        chromium_deps.vars.update(skia_deps.vars)

        if 'src/third_party/depot_tools' in chromium_deps.deps:
            depot_tools_dep = chromium_deps.deps['src/third_party/depot_tools']
            if isinstance(depot_tools_dep, dict):
                depot_tools_dep = depot_tools_dep['url']
            new_depot_tools_revision = depot_tools_dep.rsplit('@', 1)[-1]
        else:
            new_depot_tools_revision = None

        if 'upstream/dev' in self.Parameters.target_branches:
            depot_tools_branch = 'master'
        else:
            depot_tools_branch = 'master-{}'.format(
                self.target_release.major_version)

        if (not self.Parameters.skip_dt_merge and
                not self.Context.merge_depot_tools_task and
                new_depot_tools_revision != old_depot_tools_revision):
            merge_depot_tools_task = BrowserMergeDepotTools(
                self,
                owner=self.Parameters.owner,
                description='Merge depot_tools for chromium {}'.format(
                    self.target_release.version),
                notifications=self.Parameters.notifications,
                old_commit=old_depot_tools_revision,
                target_commit=new_depot_tools_revision,
                target_branch=depot_tools_branch,
                upstream_project=self.Parameters.chromium_project,
                upstream_repo=self.Parameters.chromium_dt_repo,
                fork_project=self.Parameters.snapshots_project,
                fork_repo=self.Parameters.dt_repo,
                reviewers=self.Parameters.dt_reviewers,
                use_test_bitbucket=self.Parameters.use_test_bitbucket,
                robot_login=self.Parameters.robot_login,
                robot_password_vault=self.Parameters.robot_password_vault,
                robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault)
            self.Context.merge_depot_tools_task = merge_depot_tools_task.id
            merge_depot_tools_task.enqueue()

        self.debug_breakpoint('After initial sync & gclient flatten')
        self.cleanup_repository()
        self.checkout_src()
        filtered_deps = chromium_deps.filter_out_excluded()
        self.write_deps_upstream(filtered_deps)
        self.configure_gclient_for_chromium('.DEPS.upstream')
        self.sync_chromium()
        self.debug_breakpoint('After .DEPS.upstream sync')

        self.remove_vcs_dirs()
        self.configure_gclient_for_snapshot()
        self.create_deps_snapshot(chromium_deps)
        if self.restore_gitattributes():
            self.normalize_line_endings()
        self.save_snapshot_revision_file()
        self.debug_breakpoint('Before committing snapshot')
        if os.path.exists(os.path.join(self.repo_path, '.cipd')):
            logging.error(
                '.cipd directory found in snapshot!')
            raise sandbox.common.errors.TaskFailure(
                '.cipd directory found in snapshot!')
        self.commit_snapshot()
        self.push_snapshot()
        self.Context.snapshot_pushed = True

    def on_execute(self):
        if not self.Context.snapshot_pushed:
            try:
                self.make_snapshot()
            finally:
                if self.Parameters.suspend_after_snapshot:
                    self.suspend()

        if self.Context.merge_depot_tools_task:
            self.wait_task_by_id(self.Context.merge_depot_tools_task)

    @property
    @common.utils.singleton
    def target_release(self):
        target_release = Release(self.Parameters.version, None, set())
        assert (
            target_release.upstream_branch in self.Parameters.target_branches)
        return target_release

    def get_current_release(self):
        '''
        Parse contents of src/chrome/VERSION and return Release object.
        '''

        with open(os.path.join(
                self.repo_path, 'src', 'chrome', 'VERSION')) as f:
            return Release.fromversionfile(f.read(), set())

    @action('Clone chromium snapshots.')
    def clone_repo(self):
        vcs_root = repositories.bitbucket_vcs_root(
            self.Parameters.snapshots_project,
            self.Parameters.snapshots_repo,
            testing=self.Parameters.use_test_bitbucket)
        vcs_root.clone(self.repo_path, 'upstream/dev')

        self.remote_branches = self.fetch_remote_branches()

        if self.target_release.upstream_branch in self.remote_branches:
            self.git.fetch('origin', self.target_release.upstream_branch)
            self.git.checkout('-f', 'origin/' + self.target_release.upstream_branch,
                              '-B', self.target_release.upstream_branch)
        else:
            current_release = self.get_current_release()
            while current_release > self.target_release:
                self.git.reset('--hard', 'HEAD^')
                current_release = self.get_current_release()

        return self.git.rev_parse('HEAD').strip()

    @action('Remove all files from previous snapshot.')
    def cleanup_repository(self):
        for filename in os.listdir(self.repo_path):
            if filename != '.git':
                if os.path.isdir(os.path.join(self.repo_path, filename)):
                    shutil.rmtree(os.path.join(self.repo_path, filename))
                else:
                    os.unlink(os.path.join(self.repo_path, filename))

    @action('Configure gclient for chromium fetching.')
    def configure_gclient_for_chromium(self, depsfile, custom_vars=None):
        config = {
            'solutions': [
                {
                    'managed': False,
                    'name': 'src',
                    'url': None,
                    'custom_deps': {},
                    'custom_hooks': [],
                    'deps_file': depsfile,
                    'safesync_url': '',
                    'custom_vars': custom_vars
                },
            ],
            'cache_dir': self.gclient_cache,
        }
        self.save_gclient_config(config)

    def flatten_chromium_deps(self):
        self.gclient('flatten', '--output-deps={}'.format(
            os.path.join(self.repo_path, 'src', 'DEPS.flatten')))
        with open(os.path.join(self.repo_path, 'src', 'DEPS.flatten')) as f:
            chromium_deps_contents = f.read()
        return deps.Deps.from_string(chromium_deps_contents)

    @action('Checkout chromium/src')
    def checkout_src(self):
        vcs_root = repositories.Chromium.src(
            testing=self.Parameters.use_test_bitbucket)
        vcs_root.clone(
            os.path.join(self.repo_path, 'src'), self.target_release.version)

    @action('Fetch chromium.')
    def sync_chromium(self):
        try:
            self.gclient(
                'sync',
                '--deps=all',
                '--nohooks',
                '--with_branch_heads', '--with_tags',
                '--jobs={}'.format(self.Parameters.gclient_sync_jobs))
        except subprocess.CalledProcessError:
            raise sandbox.common.errors.TemporaryError(
                'Gclient sync failed.')

    @action('Remove all vcs directories from chromium sources.')
    def remove_vcs_dirs(self):
        dirs_to_delete = []
        for dirpath, dirnames, filenames in os.walk(self.repo_path):
            if dirpath == self.repo_path:
                continue

            for dirname in dirnames:
                if dirname in ('.svn', '.hg', '.git'):
                    dirs_to_delete.append(os.path.join(dirpath, dirname))

        for dirname in dirs_to_delete:
            shutil.rmtree(dirname)

        os.unlink(os.path.join(self.repo_path, '.gclient_entries'))

    def load_deps_snapshot(self):
        with open(os.path.join(self.repo_path, 'src', '.DEPS.snapshot')) as f:
            return deps.Deps.from_string(f.read())

    @action('Backup flattened DEPS to .DEPS.upstream.')
    def write_deps_upstream(self, chromium_deps):
        chromium_deps.save(
            os.path.join(self.repo_path, 'src', '.DEPS.upstream'))

    @action('Configure gclient for snapshot dependencies fetching.')
    def configure_gclient_for_snapshot(self):
        os.unlink(os.path.join(self.repo_path, '.gclient'))
        gclient_config = {
            'cache_dir': None,
            'solutions': [
                {
                    "name": "src",
                    # solution url for snapshot must be None to make sure
                    # that gclient will not update it.
                    "url": None,
                    "deps_file": ".DEPS.snapshot",
                    "managed": False,
                    "custom_deps": {},
                    "custom_vars": {},
                },
            ],
        }
        self.save_gclient_config(gclient_config, filename='.gclient_default')

    @action('Write .DEPS.snapshot.')
    def create_deps_snapshot(self, chromium_deps):
        # Write hooks and preserved dependencies into .DEPS.snapshot.
        # Each dependency url will be redirected to the bitbucket mirror.
        deps_snapshot = chromium_deps.deps_to_preserve().redirect_to_mirror(
            self.bb, self.Parameters.snapshots_project)
        if 'src/third_party/depot_tools' in deps_snapshot.deps:
            depot_tools_dep = deps_snapshot.deps['src/third_party/depot_tools']
            if isinstance(depot_tools_dep, dict):
                url, revision = depot_tools_dep['url'].rsplit('@', 1)
            else:
                url, revision = depot_tools_dep.rsplit('@', 1)
            revision = 'upstream-merge/' + revision
            if isinstance(depot_tools_dep, dict):
                depot_tools_dep['url'] = '@'.join((url, revision))
            else:
                depot_tools_dep = '@'.join((url, revision))
            deps_snapshot.deps['src/third_party/depot_tools'] = depot_tools_dep
        deps_snapshot.save(
            os.path.join(self.repo_path, 'src', '.DEPS.snapshot'))

    def restore_gitattributes(self):
        try:
            self.git.checkout('--', '.gitattributes')
            return True
        except subprocess.CalledProcessError:
            logging.warning('There are no .gitattributes in the repository.')
            return False

    @action('Commit chromium snapshot.')
    def commit_snapshot(self):
        self.git.add('-f', '-A')
        self.git.commit(
            '-m', 'chromium-' + self.target_release.version,
            '--author=chromium <chrome-release@ya.ru>')

    @action('Push to bitbucket.')
    def push_snapshot(self):
        push_args = [
            'HEAD:refs/heads/' + branch
            for branch in self.Parameters.target_branches
        ]
        push_args.append('HEAD:refs/tags/' + self.target_release.version)
        with sandbox.sdk2.ssh.Key(
                self, self.Parameters.robot_ssh_key_vault, None):
            self.git.push('origin', *push_args)

    def save_gclient_config(self, config, filename='.gclient'):
        content = python_seriallization.format_as_python(
            config, indent=' ' * 2, pretty=True)
        logging.info('Saving gclient config:\n%s', content)
        with open(os.path.join(self.repo_path, filename), 'w') as f:
            f.write(content)

    @action('Normalizing line endings according to .gitattributes.')
    def normalize_line_endings(self):
        all_files = []
        for dirpath, dirs, files in os.walk(self.repo_path):
            if '.git' in dirs:
                dirs.remove('.git')
            all_files.extend(
                os.path.relpath(os.path.join(dirpath, filename), self.repo_path)
                for filename in files)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger('git')) as pl:
            pl.logger.propagate = 1
            check_attr = subprocess.Popen(
                ['git', 'check-attr', 'text', 'eol', '--stdin', '-z', '--'],
                cwd=self.repo_path, stdin=subprocess.PIPE,
                stdout=subprocess.PIPE, stderr=pl.stdout)

            attributes = collections.defaultdict(
                lambda: collections.defaultdict(lambda: None))

            stdout, _ = check_attr.communicate('\0'.join(all_files))
            parts = stdout.rstrip('\0').split('\0')
            while parts:
                path, attribute, value = parts[:3]
                del parts[:3]
                attributes[path][attribute] = value

        for path, attrs in attributes.iteritems():
            path_in_repo = os.path.join(self.repo_path, path)
            if os.path.islink(path_in_repo) and not os.path.exists(path_in_repo):
                logging.info('Symlink %s points to a nonexistent file %s, skipping',
                             path, os.path.realpath(path_in_repo))
                continue
            if attrs['text'] == 'set':
                target_le_name = attrs['eol'] or 'lf'
                target_le = {
                    'cr': '\r',
                    'lf': '\n',
                    'crlf': '\r\n',
                }.get(target_le_name, '\n')
                with open(path_in_repo, 'rU') as f:
                    content = f.read()
                    if f.newlines in (None, target_le):
                        continue
                logging.info('Replacing line endings in %s to %s.', path, target_le_name)
                if target_le != '\n':
                    content.replace('\n', target_le)

                with open(path_in_repo, 'wb') as f:
                    f.write(content)

    def save_snapshot_revision_file(self):
        snapshot_revision = self.bb.get_tag(self.Parameters.chromium_project, self.Parameters.chromium_repo,
                                            self.target_release.version)['latestCommit']
        logging.debug('Snapshot revision: %s', snapshot_revision)
        change_version_revision = next(self.bb.get_commits(
            self.Parameters.chromium_project, self.Parameters.chromium_repo,
            path='chrome/VERSION', until=snapshot_revision))['id']
        with open(os.path.join(self.repo_path, SNAPSHOT_REVISION_FILE), 'w') as f:
            f.write('LASTCHANGE={}\n'.format(change_version_revision))
