import os
import contextlib

from sandbox import sdk2
from sandbox.projects.browser.common import contextmanagers
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.common import teamcity


class BrowserMergeAction(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 10 * 1024  # 10GB

    class Parameters(RunFromYinTask.Parameters):
        dry_run = sdk2.parameters.Bool('Dry run')
        with sdk2.parameters.String('Action type') as action:
            action.values.start_merge = None
            action.values.start_stabilization = None
            action.values.merge_master = None
            action.values.integration_test = None
            action.values.generate_test_reference = None

        version = sdk2.parameters.String('Version for merge')
        ours_branch = sdk2.parameters.String(
            'Ours branch to merge', default='master-next')
        merge_branch = sdk2.parameters.String('Merge branch')
        test_reference_branch = sdk2.parameters.String(
            'Reference branch for merge test.')
        test_start_date = sdk2.parameters.String(
            'Start date for merge test.')

        with action.value['start_merge']:
            duties = sdk2.parameters.Dict(
                'File owners and duties for this merge',
                default={
                    'desktop': 'anata-m',
                    'android': 'uniqum',
                    'ios': 'kuznetsovs',
                    'infra': 'anata-m',
                }
            )

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff(
                'Robot login', default='robot-bro-merge')
            robot_ssh_key = sdk2.parameters.String(
                'Vault item with robot ssh key',
                default='robot-bro-merge_ssh_key')
            robot_token = sdk2.parameters.String(
                'Vault item with robot token',
                default='robot-bro-merge_token')

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @property
    def output_path(self):
        return str(self.artifacts_path('fail.diff'))

    def artifacts_path(self, *args):
        return self.path('artifacts', *args)

    @contextlib.contextmanager
    def publish_artifacts(self):
        os.makedirs(str(self.artifacts_path()))
        try:
            yield
        finally:
            if os.listdir(str(self.artifacts_path())):
                artifacts_resource = teamcity.TeamcityArtifacts(
                    self, 'Teamcity artifacts', self.artifacts_path())
                sdk2.ResourceData(artifacts_resource).ready()

    def on_execute(self):
        checkout_branch = None
        # Prepare script arguments depending on action
        args = [self.browser_path]

        if self.Parameters.action == 'start_merge':
            args += [
                self.Parameters.version,
            ]
            args += ['--duty-dict']
            args += [
                '{platform}={assignee}'.format(platform=k, assignee=v)
                for k, v in self.Parameters.duties.iteritems()
            ]
            checkout_branch = 'master-next'

        if self.Parameters.action == 'merge_master':
            args += [
                self.Parameters.merge_branch,
            ]
            checkout_branch = 'master-next'

        if self.Parameters.action == 'start_stabilization':
            args += [
                self.Parameters.ours_branch,
            ]
            if self.Parameters.version:
                args += [
                    '--version',
                    self.Parameters.version,
                ]
            checkout_branch = self.Parameters.ours_branch

        if self.Parameters.action == 'integration_test':
            args += [
                self.Parameters.ours_branch,
                self.Parameters.version,
                self.Parameters.test_reference_branch,
                '--start-date', self.Parameters.test_start_date,
                '--output', self.output_path,
            ]
            checkout_branch = self.Parameters.ours_branch

        if self.Parameters.action == 'generate_test_reference':
            args += [
                self.Parameters.ours_branch,
                self.Parameters.version,
                self.Parameters.test_reference_branch,
                '--start-date', self.Parameters.test_start_date,
            ]
            checkout_branch = self.Parameters.ours_branch

        if self.Parameters.dry_run:
            args.append('--dry-run')

        # Provide credentials if required.
        env = {}
        if self.Parameters.action in (
                'start_merge', 'start_stabilization', 'merge_master'):
            args += [
                '--oauth-login', self.Parameters.robot_login,
                '--oauth-token-env', 'ROBOT_TOKEN',
                '--oauth-no-save',
            ]
            env.update(
                ROBOT_TOKEN=sdk2.Vault.data(self.Parameters.robot_token))

        # Make browser repository
        vcs_root = repositories.Stardust.browser()
        vcs_root.update_cache_repo()
        vcs_root.clone(self.browser_path, branch=checkout_branch, config={
            'user.name': 'robot-bro-merge',
            'user.email': 'robot-bro-merge@yandex-team.ru',
        })

        with contextmanagers.ExitStack() as stack:
            # Publish artifacts on exit
            stack.enter_context(self.publish_artifacts())

            # Provide ssh key if required
            if (not self.Parameters.dry_run and
                    self.Parameters.action != 'integration_test'):
                stack.enter_context(sdk2.ssh.Key(
                    self, self.Parameters.robot_ssh_key, None))

            # Run script
            self.run_yin_script(
                'yin.merge.run.{}'.format(self.Parameters.action), args,
                extra_env=env)
