import json
import logging

import requests
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.teamcity import run_teamcity_build
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds

COLLECT_CONFLICTS_BUILD_TYPE = 'Browser_Merge_CollectConflicts'
POUR_TO_1_BUILD_TYPE = 'Browser_Merge_PourTo1'
SMART_VETO_ID = 169


class BrowserMergeAutoPourTo1(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 5 * 1024
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 1
        environments = [
            PipEnvironment('teamcity-client==6.2.1'),
            PipEnvironment('bitbucket-server-client==4.1.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String('Branch in browser repo')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff(
                'Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String(
                'Vault item with robot token',
                default='robot-bro-merge_token')

    class Context(sdk2.Context):
        branch = None
        branch_prefix = None
        slash_one_branch = None

        source_conflicts_task = None
        slash_one_conflicts_task = None
        pour_branch_task = None

    @singleton_property
    def bb(self):
        import bitbucket
        return bitbucket.BitBucket(
            url=DEFAULT_BITBUCKET_URL,
            token=sdk2.Vault.data(self.Parameters.robot_token),
        )

    @singleton_property
    def tc(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=sdk2.Vault.data(self.Parameters.robot_token),
        )

    def check_open_pr(self):
        prs_to_slash_one = self.bb.projects['STARDUST'].repos['browser'].pull_requests.get_all(
            at='refs/heads/' + self.Context.slash_one_branch
        )
        for pr in prs_to_slash_one:
            if pr.title == 'Update {prefix}/1 with {branch}'.format(
                prefix=self.Context.branch_prefix, branch=self.Context.branch,
            ):
                self.set_info(
                    '<a href="{url}">PR #{id}</a> to /1 is already open'.format(
                        url=pr.web_url, id=pr.id
                    ), do_escape=False
                )
                return True
        return False

    def check_slash_one_exists(self):
        existing_branches = {
            branch.name for branch in self.bb.projects['STARDUST'].repos['browser'].branches
        }
        return self.Context.slash_one_branch in existing_branches

    def check_already_poured(self):
        for _ in self.bb.projects['STARDUST'].repos['browser'].compare_commits(
            self.Context.branch, self.Context.slash_one_branch
        ):
            logging.info('Branch {} has not been merged into /1'.format(self.Context.branch))
            return False
        self.set_info('Branch {} has already been merged into /1'.format(self.Context.branch))
        return True

    def check_conflicts(self, task_id):
        build = self.tc.builds[task_id]
        if build.has_artifact('conflicts.json'):
            conflicts = json.loads(build.download_artifact('conflicts.json')).get('conflicts')
            if conflicts:
                self.set_info('Conflicts in {} are still unresolved'.format(build.branch_name))
                logging.info('Conflicts in %s: %s', build.branch_name, conflicts)
                return True
            else:
                return False
        else:
            raise TaskFailure('conflicts.json in {} was not found'.format(build.web_url))

    def on_enqueue(self):
        self.Context.branch = self.Parameters.branch
        if self.Parameters.branch.startswith('refs/heads/'):
            self.Context.branch = self.Parameters.branch.replace('refs/heads/', '', 1)

        self.Context.branch_prefix = self.Context.branch.rsplit('/', 1)[0]
        self.Context.slash_one_branch = '{}/1'.format(self.Context.branch_prefix)
        logging.info('Branch /1: %s\n, Branch /N: %s', self.Context.slash_one_branch, self.Context.branch)

    def on_execute(self):
        with self.memoize_stage.prerun_checks:
            if not self.check_slash_one_exists():
                return

            if self.check_already_poured() or self.check_open_pr():
                return

        with self.memoize_stage.collect_conflicts:
            self.Context.source_conflicts_task = run_teamcity_build(
                self.tc,
                branch=self.Context.branch,
                build_type=COLLECT_CONFLICTS_BUILD_TYPE,
                comment="Triggered by sandbox-{}".format(
                    self.id,
                ),
                make_unique=True,
            ).id
            self.Context.slash_one_conflicts_task = run_teamcity_build(
                self.tc,
                branch=self.Context.slash_one_branch,
                build_type=COLLECT_CONFLICTS_BUILD_TYPE,
                comment="Triggered by sandbox-{}".format(
                    self.id,
                ),
                make_unique=True,
            ).id

            raise sdk2.WaitTask(
                BrowserWaitTeamcityBuilds(
                    self,
                    description='Wait for Browser_Merge_CollectConflicts',
                    notifications=self.Parameters.notifications,
                    mode='WAIT_GIVEN',
                    builds='{} {}'.format(
                        self.Context.source_conflicts_task,
                        self.Context.slash_one_conflicts_task
                    ),
                    robot_login=self.Parameters.robot_login,
                    oauth_vault=self.Parameters.robot_token,
                ).enqueue(),
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True,
            )

        with self.memoize_stage.conflicts_check:
            children = self.find(BrowserWaitTeamcityBuilds)
            if children:
                if any(child.status != ctt.Status.SUCCESS for child in children):
                    raise TaskFailure('Browser_Merge_CollectConflicts failed')

            if (self.check_conflicts(self.Context.slash_one_conflicts_task) or
                    self.check_conflicts(self.Context.source_conflicts_task)):
                return

        with self.memoize_stage.pour_branch:
            self.Context.pour_branch_task = run_teamcity_build(
                self.tc,
                branch=self.Context.branch,
                build_type=POUR_TO_1_BUILD_TYPE,
                comment="Triggered by sandbox-{}".format(
                    self.id,
                ),
                make_unique=True,
            ).id

            raise sdk2.WaitTask(
                BrowserWaitTeamcityBuilds(
                    self,
                    description='Wait for Browser_Merge_PourTo1',
                    notifications=self.Parameters.notifications,
                    mode='WAIT_GIVEN',
                    builds=str(self.Context.pour_branch_task),
                    robot_login=self.Parameters.robot_login,
                    oauth_vault=self.Parameters.robot_token,
                ).enqueue(),
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True,
            )
        child = self.find().first()
        if child.status != ctt.Status.SUCCESS:
            raise TaskFailure('Browser_Merge_PourTo1 failed')

        pour_branch_build = self.tc.builds[self.Context.pour_branch_task]
        logging.info('pour_branch_build: %s', pour_branch_build.web_url)
        if pour_branch_build.has_artifact('update_pr_info.json'):
            pr_id = json.loads(pour_branch_build.download_artifact('update_pr_info.json')).get('pr_id')
            if pr_id:
                pr_object = self.bb.projects['STARDUST'].repos['browser'].pull_requests[pr_id]
                self.set_info('Created <a href="{url}">PR #{id}</a> to /1'.format(
                    url=pr_object.web_url, id=pr_id
                    ), do_escape=False)
                try:
                    response_json = pr_object.skip_veto(SMART_VETO_ID)
                except requests.exceptions.HTTPError as e:
                    self.set_info('Unable to skip veto due to an exception')
                    logging.exception(e)
                    raise TaskFailure
                else:
                    logging.debug(response_json)
            else:
                raise TaskFailure("update_pr_info.json in build {} is invalid. "
                                  "That really shouldn't've happened...".format(pour_branch_build))
        else:
            raise TaskFailure("Unable to find artifact update_pr_info.json in build {}"
                              "That really shouldn't've happened...".format(pour_branch_build))
