from collections import OrderedDict
import itertools
import logging
import os
import pprint
import re
import textwrap
import time

from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link, singleton

from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.merge.common import BUILDTYPE_PLATFORM, get_broken_tests, PLATFORM_CONFIG, BlacklistsDiff
from sandbox.projects.browser.merge.grupper.BrowserMergeScatterTests import BrowserMergeScatterTests
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


def should_skip_binary(binary, binary_config):
    if not binary_config.get('enabled', True):
        return True, 'Skipping disabled binary {}'.format(binary)

    if binary_config.get('tags'):
        return True, 'Skipping binary {} with tags'.format(binary)

    if binary_config.get('blacklist', None) is None:
        return True, 'Skipping binary {} as it has no blacklist'.format(binary)

    return False, None


def should_skip_test(test_name, skip_wildcards):
    if skip_wildcards and '*' in test_name:
        return True, 'Skipping blacklist entry "{}" as it contains wildcards'.format(skip_wildcards)

    return False, None


class BrowserMergeClearBlacklists(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = Tag.BROWSER & Tag.Group.LINUX  # because of teamcity and bitbucket access
        cores = 16
        ram = 32 * 1024
        environments = [
            PipEnvironment('teamcity-client==4.0.0'),
            PipEnvironment('yabrowser-blacklist-rt', version='0.0.10'),
            PipEnvironment('raven', version='5.32.0'),
            PipEnvironment('PyYAML', version='3.11'),
            GitEnvironment('2.19'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        # TODO maybe get chrome version as param and generate other params from it
        reasons_regex = sdk2.parameters.String("Regex defining which tests we're trying to enable", default=".*")
        skip_tests_with_wildcards = sdk2.parameters.Bool('Skip enabling tests with wildcard in name', default=True)
        run_builds = sdk2.parameters.Bool("Run builds instead of analyzing existing builds", default=False)
        with run_builds.value[False]:
            builds_branch = sdk2.parameters.String("Name of the branch to search latest builds")
        build_counts = sdk2.parameters.Dict(
            "Builds to analyze", default={
                'Browser_Tests_UnitTestsLinuxGn': 0,
                'Browser_Tests_UnitTestsWinGn': 0,
                'Browser_Tests_UnitTestsAndroid': 0,
                'Browser_Tests_UnitTestsMac': 0,
                'Browser_Tests_UnitTestsIos': 0,
            })
        dependent_builds_count = sdk2.parameters.Integer(
            'Limit for snapshot dependencies count', default=0)
        broken_test_threshold = sdk2.parameters.Integer("Ignore tests failing less than N%", default=35)
        pr_branch_prefix = sdk2.parameters.String("Prefix of branch for PR and tests", default='wp/robots/enable_tests/', required=True)
        destination_branch = sdk2.parameters.String("Destination branch for PR", default='master-next')
        reviewers = sdk2.parameters.List("Reviewers")
        scatter_afterwards = sdk2.parameters.Bool(
            "Launch BROWSER_MERGE_SCATTER_TESTS after this task", default=True
        )
        with sdk2.parameters.Group('Bitbucket settings'):
            project = sdk2.parameters.String(
                'project', default='STARDUST')
            browser_repo = sdk2.parameters.String(
                'repository', default='browser')
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_login = sdk2.parameters.String("Login for git commits", default="robot-bro-merge")
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity & bitbucket',
                                                 default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String("Vault item with ssh key for bitbucket",
                                                         default="robot-bro-merge_ssh_key")

    class Context(sdk2.Context):
        builds = []
        pr_url = None
        reasons_regex = ''
        pr_branch = ''

    def on_create(self):
        if not self.Parameters.reviewers:
            self.Parameters.reviewers = [self.author]

        return super(BrowserMergeClearBlacklists, self).on_create()

    @property
    @singleton
    def teamcity(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=sdk2.Vault.data(self.Parameters.oauth_vault)
        )

    @property
    @singleton
    def bitbucket(self):
        return BitBucket(
            TESTING_BITBUCKET_URL if self.Parameters.use_test_bitbucket else DEFAULT_BITBUCKET_URL,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.oauth_vault))

    @property
    @singleton
    def repo_path(self):
        return str(self.path('browser'))

    @property
    @singleton
    def git(self):
        return GitCli(
            self.repo_path,
            {
                'user.name': '{}'.format(self.Parameters.robot_login),
                'user.email': '{}@yandex-team.ru'.format(self.Parameters.robot_login),
            }
        )

    def commit_and_push_tests(self):
        try:
            self.git.checkout(self.Context.pr_branch)
            self.git.commit(
                '-a', '-m',
                textwrap.dedent(
                    '''\
                    Enable tests with reason: {}.

                    Commited by sandbox task: {}.
                    '''
                ).format(self.Parameters.reasons_regex, get_task_link(self.id)),
            )
            with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                self.git.push('--force', 'origin', self.Context.pr_branch)
        except subprocess.CalledProcessError:
            raise TaskFailure('Failed to commit or push changes')

    def enable_tests_with_reasons(self, build_types):
        self.set_info('Enabling tests by regex {}'.format(self.Context.reasons_regex))
        reasons_pattern = re.compile(self.Context.reasons_regex)
        import blacklist_rt

        for build_type in build_types:
            platform = BUILDTYPE_PLATFORM[build_type]
            logging.info('Processing build type %s', build_type)
            config_path = os.path.join(self.repo_path, PLATFORM_CONFIG[platform])
            with open(config_path) as f:
                data, formatting = blacklist_rt.load(f)

            for binary, binary_config in data.iteritems():
                skip_binary, message = should_skip_binary(binary, binary_config)
                if skip_binary:
                    logging.info(message)
                    continue

                for test_name, reason in binary_config['blacklist'].iteritems():
                    skip_test, message = should_skip_test(test_name, self.Parameters.skip_tests_with_wildcards)
                    if skip_test:
                        logging.info(message)
                        continue

                    if reasons_pattern.match(reason):
                        logging.info('Pre-enabling %s with the reason %s', test_name, reason)
                        del data[binary]['blacklist'][test_name]

            with open(config_path, 'w') as f:
                blacklist_rt.dump(f, data, formatting)

    def disable_tests_save_reason(self, build_types, broken_tests):
        logging.info('Disabling failed tests')
        reasons_pattern = re.compile(self.Context.reasons_regex)
        import blacklist_rt

        def ensure_dict(_data, _formatting, key):
            if _data.get(key) is None:
                _data[key] = OrderedDict()

            return _formatting.setdefault(
                key, blacklist_rt.Formatting())

        self.git.reset('--hard', 'HEAD~')

        for build_type in build_types:
            logging.info('Processing build type %s', build_type)
            platform = BUILDTYPE_PLATFORM[build_type]
            config_path = os.path.join(self.repo_path, PLATFORM_CONFIG[platform])

            with open(config_path) as f:
                data, formatting = blacklist_rt.load(f)

            broken_tests_on_platform = broken_tests[platform]
            for binary, binary_config in data.iteritems():
                if not binary_config.get('enabled', True):
                    continue
                broken_binary_tests = set(t.split('.', 1)[1]
                                          for t in broken_tests_on_platform
                                          if t.split('.', 1)[0] == binary)
                ensure_dict(data, formatting, binary)

                skip_binary, message = should_skip_binary(binary, binary_config)
                if skip_binary:
                    logging.info(message)
                    continue

                for test_name, reason in binary_config['blacklist'].iteritems():
                    skip_test, message = should_skip_test(test_name, self.Parameters.skip_tests_with_wildcards)
                    if skip_test:
                        logging.info(message)
                        continue

                    if reasons_pattern.match(reason) and test_name not in broken_binary_tests:
                        logging.info('Enabling %s with the reason %s', test_name, reason)
                        del data[binary]['blacklist'][test_name]

            with open(config_path, 'w') as f:
                blacklist_rt.dump(f, data, formatting)

    def on_enqueue(self):
        if not self.Context.pr_branch:
            self.Context.pr_branch = (
                self.Parameters.pr_branch_prefix + str(int(time.time() * 100)))

    def on_execute(self):
        self.Context.reasons_regex = self.Parameters.reasons_regex
        if not self.Context.reasons_regex.endswith('$'):
            self.Context.reasons_regex += '$'

        present_build_types = [build_type for build_type, count in self.Parameters.build_counts.iteritems()
                               if int(count) > 0]
        with self.memoize_stage.create_branch:
            self.bitbucket.create_branch(self.Parameters.project, self.Parameters.browser_repo,
                                         self.Context.pr_branch, self.Parameters.destination_branch)

        vcs_root = repositories.bitbucket_vcs_root(self.Parameters.project, self.Parameters.browser_repo,
                                                   testing=self.Parameters.use_test_bitbucket)
        vcs_root.clone(
            self.repo_path, self.Context.pr_branch,
            sparse_checkout_paths=['src/build/yandex/ci/configs']
        )

        with self.memoize_stage.enable_tests_before_launch:
            self.enable_tests_with_reasons(build_types=present_build_types)
            self.commit_and_push_tests()

        with self.memoize_stage.trigger_builds:
            if self.Parameters.run_builds:
                raise sdk2.WaitTask(
                    BrowserWaitTeamcityBuilds(
                        self,
                        description='Check builds in merge branch',
                        notifications=self.Parameters.notifications,
                        mode='RUN_NEW',
                        kill_on_cancel=True,
                        dependent_builds_count=self.Parameters.dependent_builds_count,
                        branch=self.Context.pr_branch,
                        build_counts=self.Parameters.build_counts,
                        oauth_vault=self.Parameters.oauth_vault,
                    ).enqueue(),
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    True,
                )

        if self.Parameters.run_builds:
            wait_builds_task = list(self.find())[0]
            if wait_builds_task.status not in Status.Group.SUCCEED:
                raise TaskFailure('Child task BrowserWaitTeamcityBuilds unexpectedly failed')
            self.Context.builds = wait_builds_task.Context.finished_builds
        else:
            self.Context.builds = [
                b['id']
                for build_type, count in self.Parameters.build_counts.iteritems()
                for b in self.teamcity.rest_api.builds.get(
                    locator=dict(buildType__id=build_type, branch=self.Parameters.builds_branch),
                    count=count, fields='build(id)',
                ).get('build', [])
            ]

        broken_tests = get_broken_tests(self.Context.builds, self.teamcity, self.Parameters.build_counts,
                                        self.Parameters.broken_test_threshold)
        logging.info('Broken tests: \n%s', pprint.pformat(broken_tests))

        self.disable_tests_save_reason(build_types=present_build_types, broken_tests=broken_tests)

        changes = self.git.status('--porcelain')
        if not changes:
            self.set_info('No changes in blacklists were made')
            return

        diff = self.git.diff()
        with open(str(self.path('diff')), 'w') as f:
            f.write(diff)
        errors_resource = BlacklistsDiff(self, 'Diff of blacklist changes', self.path('diff'))
        sdk2.ResourceData(errors_resource).ready()

        self.set_info('git diff of blacklist changes is in resources')

        self.commit_and_push_tests()

        resp = self.bitbucket.create_pr(
            self.Parameters.project, self.Parameters.browser_repo,
            'Enable tests with reasons: /{}/'.format(self.Context.reasons_regex),
            'Generated automatically by {}'.format(get_task_link(self.id)),
            self.Context.pr_branch,
            self.Parameters.destination_branch,
            self.Parameters.reviewers or []
        ).response

        self.Context.pr_url = resp['links']['self'][0]['href']
        self.set_info('<a href="{}">PR to {}</a> created successfully'.format(
            self.Context.pr_url, self.Parameters.destination_branch,
        ), do_escape=False)

        if self.Parameters.scatter_afterwards:
            scatter_task = BrowserMergeScatterTests(
                self,
                description='Scatter tests in branch {}'.format(self.Context.pr_branch),
                yin_branch='master',
                tc_branch=self.Context.pr_branch if self.Parameters.run_builds else self.Parameters.builds_branch,
                checkout_branch=self.Parameters.destination_branch,
                num_builds=max(self.Parameters.build_counts.values()),
                reset_owners=False,
                ignore_blacklists=False,
                drop_binaries=False,
                version=self.Parameters.reasons_regex,
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.oauth_vault,
            )
            self.set_info(
                '<a href={}>BROWSER_MERGE_SCATTER_TESTS launched</a>'.format(get_task_link(scatter_task.id)),
                do_escape=False
            )

            scatter_task.enqueue()

    @sdk2.footer()
    def footer(self):
        footer = []

        if self.Context.pr_url is not None:
            footer.append({
                'helperName': '',
                'content': '<h3><a href="{}">PR to {}</a></h3>'.format(
                    self.Context.pr_url, self.Parameters.destination_branch,
                ),
            })

        footer.extend(itertools.chain.from_iterable(filter(None, (task.footer() for task in self.find()))))
        return footer

    @sdk2.header()
    def header(self):
        return list(itertools.chain.from_iterable(filter(None, (task.header() for task in self.find()))))
