# coding=utf-8
import logging
import os

import jinja2

from sandbox.projects.browser.merge.BrowserMergeBase import BrowserMergeBase
from sandbox.projects.browser.merge.common import check_merge_in_progress, get_branch_lock
from sandbox.projects.browser.common.mail import send_email
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.common.bitbucket import READ_ONLY, BRANCH
from sandbox import sdk2


BITBUCKET_ACCESS_KEY_ID = 17080


class BrowserMergeCloseMaster(BrowserMergeBase):
    class Requirements(sdk2.Requirements):
        disk_space = 100

    class Parameters(BrowserMergeBase.Parameters):
        robot_login = BrowserMergeBase.Parameters.robot_login(ui=None)
        robot_ssh_key_vault = BrowserMergeBase.Parameters.robot_ssh_key_vault(ui=None)
        tests_branch = sdk2.parameters.String('Tests branch', required=True)
        compilation_branch = sdk2.parameters.String('Compilation branch', required=True)
        merge_version = sdk2.parameters.String('Merge version', required=True)
        merge_sheriffs = sdk2.parameters.List(
            'Merge sheriffs', default=[],
            description='These people will be able to force merge pr to merge tests branch')

    def get_email_text(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        text = env.get_template('master_closed.jinja').render(
            tests_branch=self.Parameters.tests_branch,
            compilation_branch=self.Parameters.compilation_branch,
            version=self.Parameters.merge_version,
        )
        logging.debug('Email text %s', text)
        return text

    def restrict_branch(self, branch, users=None, accessKeys=None):
        self.bb.create_restriction(
            'stardust', 'browser',
            restriction_type=READ_ONLY,
            matcher_type=BRANCH,
            matcher='refs/heads/' + branch,
            accessKeys=accessKeys,
            users=users,
        )

    def on_create(self):
        if not self.Parameters.merge_sheriffs:
            self.Parameters.merge_sheriffs = [self.author]

    def on_execute(self):
        if get_branch_lock(self.bb, 'master-next'):
            raise TaskFailure('master-next is already closed')
        if not check_merge_in_progress(self.bb, [self.Parameters.tests_branch, self.Parameters.compilation_branch]):
            raise TaskFailure('Merge is not in progress')
        self.restrict_branch('master-next', accessKeys=[BITBUCKET_ACCESS_KEY_ID])
        self.restrict_branch(self.Parameters.tests_branch, users=self.Parameters.merge_sheriffs)
        send_email(
            sender_email='robot-bro-merge@yandex-team.ru',
            recipients=['browser-dev@yandex-team.ru',
                        'browser-dev-announces@yandex-team.ru',
                        'browser-merge@yandex-team.ru'],
            subject=u'[Merge] master-next закрыт',
            text=self.get_email_text(),
            reply_to=','.join([
                'hpotter@yandex-team.ru',
                'browser-merge@yandex-team.ru',
            ]),
            headers={
                'X-Browser-Merge': 'yes',
                'X-Browser-Merge-Version': self.Parameters.merge_version,
                'X-Browser-Merge-Type': 'master-closed',
            },
            dry_run=self.Parameters.dry_run,
        )
