import os
import subprocess

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.common import teamcity


class BrowserMergeCollectConflicts(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 10 * 1024  # 10GB

    class Parameters(RunFromYinTask.Parameters):
        branch = sdk2.parameters.String('Branch in browser repo')
        commit = sdk2.parameters.String('Commit in browser repo')
        conflicts_branch = sdk2.parameters.String(
            'Merge branch with conflicts')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff(
                'Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String(
                'Vault item with robot token', default='robot-bro-merge_token')

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @property
    def git(self):
        return GitCli(self.browser_path)

    @property
    def output_dir_path(self):
        return str(self.path('output_dir'))

    @property
    def output_path(self):
        return os.path.join(self.output_dir_path, 'conflicts.json')

    def on_execute(self):
        try:
            vcs_root = repositories.Stardust.browser()
            vcs_root.update_cache_repo(
                self.Parameters.branch, 'refs/heads/master-next')
            vcs_root.clone_bare(self.browser_path)

            branch = self.Parameters.branch
            if branch.startswith('refs/heads/'):
                branch = branch.split('/', 2)[-1]
            self.git.fetch('origin', 'master-next',
                           '+refs/heads/{0}:refs/heads/{0}'.format(branch))
        except subprocess.CalledProcessError:
            raise errors.TemporaryError('Checkout error')

        os.mkdir(self.output_dir_path)

        args = [
            branch, self.browser_path, self.output_path,
            '--oauth-login', self.Parameters.robot_login,
            '--oauth-token-env', 'ROBOT_TOKEN',
            '--oauth-no-save', '-vvv',
        ]
        # if self.Parameters.conflicts_branch == self.Parameters.branch:
        #     args.append('--report-teamcity-statistics')

        try:
            self.run_yin_script(
                'yin.merge.reports.conflicts', args,
                extra_env=dict(
                    ROBOT_TOKEN=sdk2.Vault.data(self.Parameters.robot_token)))
        finally:
            if os.listdir(self.output_dir_path):
                artifacts_resource = teamcity.TeamcityArtifacts(
                    self, 'Teamcity artifacts', self.output_dir_path)
                sdk2.ResourceData(artifacts_resource).ready()
