'''
Author: Aleksandr Derbenev <alex-ac@yandex-team.ru>
'''

import time

from sandbox.common.types.client import Tag
from sandbox.common.utils import get_task_link, singleton_property

from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli

from sandbox import sdk2


class BrowserMergeDepotTools(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = Tag.BROWSER & Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024
        environments = (
            GitEnvironment('2.19'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        pr_branch = sdk2.parameters.String('Branch for PR', required=False)
        old_commit = sdk2.parameters.String('Previous commit to make branch from', required=False)
        target_commit = sdk2.parameters.String('Commit to merge', required=True)
        target_branch = sdk2.parameters.String('Target branch for PR', required=True)
        reviewers = sdk2.parameters.List('Reviewers')
        with sdk2.parameters.Group('Bitbucket settings'):
            upstream_project = sdk2.parameters.String(
                'Upstream project', default='CHROMIUM')
            upstream_repo = sdk2.parameters.String(
                'Upstream repo', default='tools-depot_tools')
            fork_project = sdk2.parameters.String(
                'Fork project', default='STARDUST')
            fork_repo = sdk2.parameters.String(
                'Fork repo', default='depot_tools')
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.String(
                'Login for bitbucket.',
                default='robot-bro-merge')
            robot_password_vault = sdk2.parameters.String(
                'Vault item with password for bitbucket.',
                default='robot-bro-merge_password')
            robot_ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key for bitbucket.',
                default='robot-bro-merge_ssh_key')

    def default_pr_branch(self):
        return 'users/{}/update/{}'.format(
            self.Parameters.robot_login, int(time.time() * 1000),
        )

    def on_create(self):
        if not self.Parameters.pr_branch:
            self.Parameters.pr_branch = self.default_pr_branch()

        if not self.Parameters.reviewers:
            self.Parameters.reviewers = [self.author]

        return super(BrowserMergeDepotTools, self).on_create()

    def get_bitbucket_session(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(
            bitbucket_url,
            self.Parameters.robot_login,
            sdk2.Vault.data(self.Parameters.robot_password_vault))

    @property
    def repo_path(self):
        return str(self.path('depot_tools'))

    @singleton_property
    def git(self):
        return GitCli(self.repo_path)

    def on_execute(self):
        bb = self.get_bitbucket_session()
        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            vcs_root = repositories.bitbucket_vcs_root(
                self.Parameters.upstream_project,
                self.Parameters.upstream_repo,
                testing=self.Parameters.use_test_bitbucket)
            vcs_root.clone(self.repo_path, 'main')
            fork_url = bb.get_repo_clone_url(
                self.Parameters.fork_project,
                self.Parameters.fork_repo)
            pr_branch = self.Parameters.pr_branch or self.default_pr_branch()
            self.git.remote('add', 'fork', fork_url)
            self.git.push('fork', '{}:refs/heads/{}'.format(self.Parameters.target_commit, pr_branch))
            remote_branches = [
                line.split('\t', 1)[-1]
                for line in sorted(self.git.ls_remote('fork').strip().splitlines())
            ]
            remote_tags = [
                line.split('\t', 1)[-1]
                for line in sorted(self.git.ls_remote(
                    'fork',
                    'refs/tags/upstream-merge/*').strip().splitlines())]
            if 'refs/heads/' + self.Parameters.target_branch not in remote_branches:
                if (self.Parameters.old_commit and
                        'refs/heads/upstream-merge/{}'.format(
                            self.Parameters.old_commit) in remote_tags):
                    self.git.fetch('fork')
                    self.git.push('fork', 'fork/upstream-merge/{}:refs/heads/{}'.format(
                        self.Parameters.old_commit, self.Parameters.target_branch))
                else:
                    self.git.fetch('fork', 'master')
                    self.git.push('fork', 'fork/master:refs/heads/{}'.format(
                        self.Parameters.target_branch))
            self.git.fetch('fork', self.Parameters.target_branch)
            self.git.push('fork', 'fork/{}:refs/heads/upstream-merge/{}'.format(
                self.Parameters.target_branch, self.Parameters.target_commit))

        pr = bb.create_pr(
            self.Parameters.fork_project, self.Parameters.fork_repo,
            title='Update depot_tools with upstream.',
            description='Generated automatically by {}.'.format(
                get_task_link(self.id)),
            from_ref=pr_branch,
            to_ref=self.Parameters.target_branch,
            reviewers=self.Parameters.reviewers)
        self.set_info('<a href="{}">PR #{}</a> created successfully'.format(
            pr.web_url, pr.id), do_escape=False)
