from collections import OrderedDict
import itertools
import logging
import pprint
import time

from sandbox import common

from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link

from sandbox.projects.browser.common import bitbucket as bb
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.merge.common import PLATFORM_CONFIG, get_broken_tests, group_tests, BlacklistsDiff
from sandbox.projects.browser.merge.grupper.BrowserMergeScatterTests import BrowserMergeScatterTests
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


class BrowserMergeDisableBrokenTests(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = Tag.BROWSER & Tag.Group.LINUX  # because of teamcity and bitbucket access
        cores = 16
        ram = 32 * 1024
        environments = [
            PipEnvironment('teamcity-client==4.0.0'),
            PipEnvironment('yabrowser-blacklist-rt'),
            PipEnvironment('raven', version='5.32.0'),
            GitEnvironment('2.19'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        # TODO maybe get chrome version as param and generate other params from it
        branch = sdk2.parameters.String("Branch", required=True)
        run_builds = sdk2.parameters.Bool("Run builds instead of analyzing existing builds", default=True)
        with run_builds.value[False]:
            builds_branch = sdk2.parameters.String("Branch to analyze builds")
        build_counts = sdk2.parameters.Dict(
            "Builds to analyze", default={
                'Browser_Tests_UnitTestsLinuxGn': 0,
                'Browser_Tests_UnitTestsLinuxStatic': 0,
                'Browser_Tests_UnitTestsWinGn': 0,
                'Browser_Tests_UnitTestsWin64': 0,
                'Browser_Tests_UnitTestsWinStatic': 0,
                'Browser_Tests_UnitTestsMac': 0,
                'Browser_Tests_UnitTestsMacStatic': 0,
                'Browser_Tests_UnitTestsAndroid': 0,
                'Browser_Tests_UnitTestsIos': 0,
            })
        dependent_builds_count = sdk2.parameters.Integer(
            'Limit for snapshot dependencies count', default=0)
        broken_test_threshold = sdk2.parameters.Integer("Ignore tests failing less than N%", default=35)
        disable_other_platforms_threshold = sdk2.parameters.Integer(
            "Disable tests if more than N% failed, and they also failed on other platforms", default=100)
        pr_branch = sdk2.parameters.String("Branch for PR", required=True)
        disable_reason = sdk2.parameters.String("Disable reason for blacklist", default='TODO')
        reviewers = sdk2.parameters.List("Reviewers")
        direct_push = sdk2.parameters.Bool("Directly push changes to branch", default=False)
        scatter_afterwards = sdk2.parameters.Bool(
            "Launch BROWSER_MERGE_SCATTER_TESTS after this task", default=True
        )

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_login = sdk2.parameters.String('Robot login', default='robot-bro-merge')
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity & bitbucket',
                                                 default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String("Vault item with ssh key for bitbucket",
                                                         default="robot-bro-merge_ssh_key")

    class Context(sdk2.Context):
        builds = []
        pr_url = None

    def on_create(self):
        if not self.Parameters.pr_branch:
            self.Parameters.pr_branch = 'users/{}/disable-tests/{}'.format(
                self.author, int(time.time() * 1000),
            )

        if not self.Parameters.reviewers:
            self.Parameters.reviewers = [self.author]

        return super(BrowserMergeDisableBrokenTests, self).on_create()

    @common.utils.singleton_property
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=sdk2.Vault.data(self.Parameters.oauth_vault)
        )

    @common.utils.singleton_property
    def bitbucket(self):
        url = bb.DEFAULT_BITBUCKET_URL
        return bb.BitBucket(url, 'x-oauth-token', sdk2.Vault.data(self.Parameters.oauth_vault))

    def disable_tests(self, config_path, tests):
        import blacklist_rt

        with open(config_path) as f:
            data, formatting = blacklist_rt.load(f)

        def ensure_dict(_data, _formatting, key):
            if _data.get(key) is None:
                _data[key] = OrderedDict()

            return _formatting.setdefault(
                key, blacklist_rt.Formatting())

        malformed_tests = {
            t for t in tests if '.' not in t
        }
        if malformed_tests:
            self.set_info('Some test names are malformed (no dot in test name): {}'.format(', '.join(malformed_tests)))

        binaries = set(t.split('.', 1)[0] for t in tests if t not in malformed_tests)
        for binary in binaries:
            ensure_dict(data, formatting, binary)
            if not formatting[binary].lines:
                # add empty line before binary entry to separate from previous one.
                formatting[binary].lines.append(None)

            ensure_dict(data[binary], formatting[binary].subkeys, 'blacklist')

            binary_tests = sorted(
                t.split('.', 1)[1] for t in tests if t not in malformed_tests and t.split('.', 1)[0] == binary
            )
            data[binary]['blacklist'].update(
                (t, self.Parameters.disable_reason) for t in binary_tests
                if not data[binary]['blacklist'].get(t)
            )
            if len(data[binary]['blacklist']) > len(binary_tests):
                # add empty line before first test to separate old and new tests
                formatting[binary].subkeys['blacklist'].subkeys[binary_tests[0]] = blacklist_rt.Formatting(lines=[None])

        with open(config_path, 'w') as f:
            blacklist_rt.dump(f, data, formatting)

    def browser_path(self, *args):
        return self.path('browser', *args)

    @common.utils.singleton_property
    def git(self):
        return GitCli(
            str(self.browser_path()),
            {
                'user.name': '{}'.format(self.Parameters.robot_login),
                'user.email': '{}@yandex-team.ru'.format(self.Parameters.robot_login),
            }
        )

    def push(self, branch):
        try:
            with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                self.git.push('origin', branch)
        except subprocess.CalledProcessError:
            logging.exception('Failed to push commit to %s', branch)
            raise TaskFailure('Failed to push commit to {}'.format(branch))

    def on_execute(self):
        with self.memoize_stage.trigger_builds:
            if self.Parameters.run_builds:
                raise sdk2.WaitTask(
                    BrowserWaitTeamcityBuilds(
                        self,
                        description='Check builds in merge branch',
                        notifications=self.Parameters.notifications,
                        mode='RUN_NEW',
                        kill_on_cancel=True,
                        dependent_builds_count=self.Parameters.dependent_builds_count,
                        branch=self.Parameters.branch,
                        build_counts=self.Parameters.build_counts,
                        oauth_vault=self.Parameters.oauth_vault,
                    ).enqueue(),
                    Status.Group.FINISH,
                    True,
                )

        if self.Parameters.run_builds:
            wait_builds_task = list(self.find())[0]
            if wait_builds_task.status not in Status.Group.FINISH:
                raise TaskFailure("Child task BrowserWaitTeamcityBuilds unexpectedly failed")

            build_ids = wait_builds_task.Context.finished_builds
        else:
            build_ids = [
                b['id']
                for build_type, count in self.Parameters.build_counts.iteritems()
                for b in self.teamcity_client.rest_api.builds.get(
                    locator=dict(buildType__id=build_type, branch=self.Parameters.builds_branch or self.Parameters.branch),
                    count=count, fields='build(id)',
                ).get('build', [])
            ]

        broken_tests = get_broken_tests(build_ids, self.teamcity_client, self.Parameters.build_counts,
                                        self.Parameters.broken_test_threshold,
                                        self.Parameters.disable_other_platforms_threshold)
        logging.info('Broken tests: \n%s', pprint.pformat(broken_tests))

        tests_for_blacklists = group_tests(broken_tests)
        logging.info('Broken tests grouped by config file: \n%s', pprint.pformat(tests_for_blacklists))

        repositories.Stardust.browser(filter_branches=False).clone(
            str(self.browser_path()), self.Parameters.branch,
        )

        for platform, platform_tests in tests_for_blacklists.iteritems():
            config_path = str(self.browser_path(PLATFORM_CONFIG[platform]))
            self.disable_tests(config_path, platform_tests)

        changes = self.git.status('--porcelain')
        if not changes:
            self.set_info('No changes in blacklists were made')
            return

        diff = self.git.diff()
        with open(str(self.path('diff')), 'w') as f:
            f.write(diff)
        errors_resource = BlacklistsDiff(self, 'Diff of blacklist changes', self.path('diff'))
        sdk2.ResourceData(errors_resource).ready()
        self.set_info('git diff of blacklist changes is in resources')

        self.git.checkout('-b', self.Parameters.pr_branch)
        self.git.commit('-a', '-m', 'Disable failing tests')
        self.push(branch=self.Parameters.pr_branch)

        try:
            pr = self.bitbucket.create_pr(
                project='stardust',
                repo='browser',
                title='Disable tests',
                description='Generated automatically by {}'.format(get_task_link(self.id)),
                from_ref=self.Parameters.pr_branch,
                to_ref=self.Parameters.branch,
                reviewers=self.Parameters.reviewers,
            )
        except bb.RequestError:
            raise TaskFailure('Failed to create PR from {} to {}'.format(
                self.Parameters.pr_branch, self.Parameters.branch,
            ))
        else:
            self.Context.pr_url = pr.web_url
            self.set_info('<a href="{}">PR to {}</a> created successfully'.format(
                self.Context.pr_url, self.Parameters.branch,
            ), do_escape=False)
            if self.Parameters.direct_push:
                self.git.checkout(self.Parameters.branch)
                self.git.merge('-m', 'Pushing blacklists directly from {}'.format(self.Parameters.pr_branch),
                               '--no-ff', '--no-edit', self.Parameters.pr_branch)
                self.push(self.Parameters.branch)

        if self.Parameters.scatter_afterwards:
            scatter_task = BrowserMergeScatterTests(
                self,
                description='Scatter tests in branch {}'.format(self.Parameters.branch),
                yin_branch='master',
                tc_branch=self.Parameters.builds_branch or self.Parameters.branch,
                checkout_branch=self.Parameters.branch,
                num_builds=max(self.Parameters.build_counts.values()),
                reset_owners=False,
                ignore_blacklists=False,
                drop_binaries=False,
                version=self.Parameters.disable_reason,
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.oauth_vault,
            )
            self.set_info(
                '<a href={}>BROWSER_MERGE_SCATTER_TESTS launched</a>'.format(get_task_link(scatter_task.id)),
                do_escape=False
            )

            scatter_task.enqueue()

    @sdk2.footer()
    def footer(self):
        footer = []

        if self.Context.pr_url is not None:
            footer.append({
                'helperName': '',
                'content': '<h3><a href="{}">PR to {}</a></h3>'.format(
                    self.Context.pr_url, self.Parameters.branch,
                ),
            })

        footer.extend(itertools.chain.from_iterable(filter(None, (task.footer() for task in self.find()))))

        return footer

    @sdk2.header()
    def header(self):
        header = list(itertools.chain.from_iterable(task.header() for task in self.find() if task))
        return header
