import itertools
import time

from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status

from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds
from sandbox.projects.browser.merge.BrowserMergeDisableBrokenTests import BrowserMergeDisableBrokenTests

from sandbox import sdk2


class BrowserMergeFinalChecks(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String("Branch", required=True)

        with sdk2.parameters.Group("Tests") as tests_group:  # TODO use params from BrowserMergeDisableBrokenTests task
            tests_build_counts = sdk2.parameters.Dict(
                'Builds to trigger to find broken tests', default={
                    'Browser_Tests_UnitTestsLinuxGn': 30,
                    'Browser_Tests_UnitTestsWinGn': 30,
                    'Browser_Tests_UnitTestsWin64': 30,
                    'Browser_Tests_UnitTestsMac': 30,
                    'Browser_Tests_UnitTestsAndroid': 30,
                    'Browser_Tests_UnitTestsIos': 30,
                })
            dependent_builds_count = sdk2.parameters.Integer(
                'Limit for snapshot dependencies count', default=10)
            disable_reason = sdk2.parameters.String("Disable reason for blacklist", default='TODO')
            broken_test_threshold = sdk2.parameters.Integer("Ignore tests failing less than N%", default=50)
            blacklist_reviewers = sdk2.parameters.List(
                'Reviewers for blacklist PR')

        with sdk2.parameters.Group("Non-tests builds") as check_group:
            check_build_counts = sdk2.parameters.Dict(
                'Builds to check', default={
                    'BrowserBrandedDistributives_BrandedCanaryLinuxNew': 3,
                    'BrowserBrandedDistributives_BrandedCanaryWin': 3,
                    'BrowserBrandedDistributives_BrandedCanaryMac': 3,

                    'BrowserBareUnsignedBuilds_BisectBuilds_BisectSnapshots_Linux': 3,
                    'BrowserBareUnsignedBuilds_BisectBuilds_BisectSnapshots_Win': 3,
                    'BrowserBareUnsignedBuilds_BisectBuilds_BisectSnapshots_Mac': 3,
                })

        with sdk2.parameters.Group("Credentials") as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity & bitbucket',
                                                 default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String("Vault item with ssh key for bitbucket",
                                                         default="robot-bro-merge_ssh_key")

    class Context(sdk2.Context):
        finished_builds = []
        running_builds = None

    def on_create(self):
        if not self.Parameters.blacklist_reviewers:
            self.Parameters.blacklist_reviewers = [self.author]

        return super(BrowserMergeFinalChecks, self).on_create()

    def on_execute(self):
        with self.memoize_stage.trigger_builds:
            if self.Parameters.branch.startswith('merge-'):
                pr_branch = 'wp/{}/disable-tests/{}'.format(
                    self.Parameters.branch.split('/')[0],
                    int(time.time() * 1000),
                )
            else:
                pr_branch = 'users/{}/disable-tests/{}'.format(
                    self.author, int(time.time() * 1000),
                )

            raise sdk2.WaitTask(
                [
                    BrowserMergeDisableBrokenTests(
                        self,
                        description='Disable broken tests in merge branch',
                        notifications=self.Parameters.notifications,
                        branch=self.Parameters.branch,
                        run_builds=True,
                        build_counts=self.Parameters.tests_build_counts,
                        dependent_builds_count=self.Parameters.dependent_builds_count,
                        broken_test_threshold=self.Parameters.broken_test_threshold,
                        pr_branch=pr_branch,
                        disable_reason=self.Parameters.disable_reason,
                        reviewers=self.Parameters.blacklist_reviewers,
                        direct_push=True,
                        scatter_afterwards=True,
                        oauth_vault=self.Parameters.oauth_vault,
                        robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault,
                    ).enqueue(),
                    BrowserWaitTeamcityBuilds(
                        self,
                        description='Check builds in merge branch',
                        notifications=self.Parameters.notifications,
                        mode='RUN_NEW',
                        branch=self.Parameters.branch,
                        build_counts=self.Parameters.check_build_counts,
                        oauth_vault=self.Parameters.oauth_vault,
                    ).enqueue(),
                ],
                Status.Group.FINISH,
                wait_all=True,
            )

        if not all(task.status == Status.SUCCESS for task in self.find()):
            raise TaskFailure('Some of child tasks failed')

    @sdk2.footer()
    def footer(self):
        return list(itertools.chain.from_iterable(filter(None, (task.footer() for task in self.find()))))

    @sdk2.header()
    def header(self):
        return list(itertools.chain.from_iterable(task.header() for task in self.find() if task))
