# coding=utf-8
import logging
import os
from subprocess import CalledProcessError

import jinja2

from sandbox.projects.browser.merge.BrowserMergeBase import BrowserMergeBase
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import decorators
from sandbox.projects.browser.common.mail import send_email
from sandbox.projects.browser.merge.common import get_branch_lock, check_merge_in_progress

CLOSE_MASTER_SSH_KEY_VAULT = 'robot-bro-merge_close_master_ssh_key'


class BrowserMergeFinish(BrowserMergeBase):

    class Parameters(BrowserMergeBase.Parameters):
        tests_branch = sdk2.parameters.String('Tests branch', required=True)
        compilation_branch = sdk2.parameters.String('Compilation branch', required=True)
        merge_version = sdk2.parameters.String('Merge version', required=True)

        comments_about_merge = sdk2.parameters.List(
            'Comments about this merge', description='What happened in this merge: new blockers, big changes etc')

    def get_email_text(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        text = env.get_template('merge_finished.jinja').render(
            version=self.Parameters.merge_version,
            comments=self.Parameters.comments_about_merge,
        )
        logging.debug('Email text %s', text)
        return text

    def merge_to_master(self):
        logging.info('Merging compilation branch to master-next')
        self.git.checkout('-f', 'origin/master-next')
        if not self.Parameters.dry_run:
            self.git.merge('--no-ff', 'origin/{}'.format(self.Parameters.compilation_branch),
                           '-m', 'Update master-next with chromium {}'.format(self.Parameters.merge_version))
            with sdk2.ssh.Key(self, CLOSE_MASTER_SSH_KEY_VAULT, None):
                self.git.push('origin', 'HEAD:master-next')

    def unlock_branches(self, master_lock, compilation_branch_lock):
        logging.info('Unlocking branches')
        self.bb.delete_restriction('stardust', 'browser', master_lock['id'])
        self.bb.delete_restriction('stardust', 'browser', compilation_branch_lock['id'])

    def is_master_head_in_compilation_branch(self):
        try:
            self.git.merge_base(
                '--is-ancestor', 'origin/master-next', 'origin/{}'.format(self.Parameters.compilation_branch))
            return True
        except CalledProcessError:
            return False

    @decorators.retries(5, delay=1, backoff=2)
    def delete_merge_pins(self):
        logging.info('Deleting merge-pins')
        self.bb.delete_merge_pins_in_branch('stardust', 'browser', 'master-next')

    def on_execute(self):
        master_lock = get_branch_lock(self.bb, 'master-next')
        tests_branch_lock = get_branch_lock(self.bb, self.Parameters.tests_branch)

        if not check_merge_in_progress(self.bb, [self.Parameters.tests_branch, self.Parameters.compilation_branch]):
            raise TaskFailure('Merge is not in progress')
        if not master_lock:
            raise TaskFailure('master-next is not closed.')

        self.clone_repo('master-next')
        self.git.fetch('origin', 'master-next', self.Parameters.compilation_branch)

        if not self.is_master_head_in_compilation_branch():
            raise TaskFailure('master-next is not poured into compilation branch.')

        self.merge_to_master()
        self.delete_merge_pins()
        self.unlock_branches(master_lock, tests_branch_lock)
        send_email(
            sender_email='robot-bro-merge@yandex-team.ru',
            recipients=[
                'browser-dev@yandex-team.ru',
                'browser-dev-announces@yandex-team.ru',
                'browser-merge@yandex-team.ru',
                'golubtsov@yandex-team.ru',
                'lafkadio@yandex-team.ru',
                'dmgor@yandex-team.ru',
                'hpotter@yandex-team.ru',
            ],
            subject=u'[Merge] master-next открыт',
            text=self.get_email_text(),
            reply_to=','.join([
                'hpotter@yandex-team.ru',
                'browser-merge@yandex-team.ru',
            ]),
            headers={
                'X-Browser-Merge': 'yes',
                'X-Browser-Merge-Version': self.Parameters.merge_version,
                'X-Browser-Merge-Type': 'master-opened',
            },
            dry_run=self.Parameters.dry_run,
        )
