# coding=utf-8
import logging
import os
import time

import six
import yaml
from sandbox import sdk2
from sandbox.common.utils import singleton_property, get_task_link

from sandbox.projects.browser.common import LXC_RESOURCE_ID
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.RunYinScript import RunYinScript
from sandbox.projects.browser.common.git.git_cli import GitCli

SCRIPT_PATH = '.'.join(['yin', 'merge', 'generate_merge_tasks'])
COMMON_TEMPLATE_PATH = os.path.join('yin', 'merge', 'daily_merge_tasks_template_common.yaml')
THREE_WEEKS_TEMPLATE_PATH = os.path.join('yin', 'merge', 'daily_merge_tasks_template_3_weeks.yaml')
FOUR_WEEKS_TEMPLATE_PATH = os.path.join('yin', 'merge', 'daily_merge_tasks_template_4_weeks.yaml')
DUTY_CONFIG_PATH = os.path.join('yin', 'sheriff', 'duty_bot', 'config', 'duty_config.yaml')
CHANNEL_NAME = 'merge'


class DutyConfigDiff(sdk2.Resource):
    ttl = 30


class BrowserMergeGenerateDutyTasks(RunYinScript):
    _PYTHON_VERSION = '3.9.7'

    class Parameters(RunYinScript.Parameters):
        reviewers = sdk2.parameters.List('Reviewers', required=True, default=['hpotter', 'anata-m'])
        chromium_version = sdk2.parameters.Integer('Chromium version')
        start_date = sdk2.parameters.StrictString(
            '(optional) Merge start date, format: YYYY-MM-DD. Default: Monday, current week',
            regexp=r"^$|^\d{4}-\d{2}-\d{2}$",
        )
        duty_config_path = sdk2.parameters.String('Path to duty-bot config',
                                                  required=True, default=DUTY_CONFIG_PATH)

        use_custom_template = sdk2.parameters.Bool('Use custom template', default=False)
        with use_custom_template.value[True]:
            custom_template_path = sdk2.parameters.String('Path to custom task template', required=True)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.String(
                'Login for teamcity & bitbucket', default='robot-bro-merge')
            robot_token = sdk2.parameters.String('Vault item with robot token',
                                                 default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String('Vault item with ssh key for bitbucket',
                                                         default='robot-bro-merge_ssh_key')

        _container = sdk2.parameters.Container('Linux container', default_value=LXC_RESOURCE_ID)

    class Context(RunYinScript.Context):
        merge_length_str = ''
        merge_start_template = ''
        merge_end_template = ''

    @property
    def intermediate_output(self):
        return self.yin_path('output.txt')

    @singleton_property
    def git(self):
        return GitCli(
            str(self.yin_path()),
            config={
                'user.name': self.Parameters.robot_login,
                'user.email': self.Parameters.robot_login + '@yandex-team.ru',
            }
        )

    @singleton_property
    def bb(self):
        return bitbucket.BitBucket(
            bitbucket.DEFAULT_BITBUCKET_URL,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.robot_token))

    def script_cmd(self, python_executable):
        return [
            str(python_executable), '-m', SCRIPT_PATH,
            '--output', str(self.intermediate_output),
            '--start-merge-template-path', self.Context.merge_start_template,
            '--end-merge-template-path', self.Context.merge_end_template,
            '--start-date', self.Parameters.start_date,
        ]

    def write_tasks_to_config(self):
        pr_branch = 'wp/robots/merge_tasks/{}'.format(int(time.time() * 1000))
        self.git.checkout('-f', '-B', pr_branch, self.Parameters.branch)
        try:
            with self.intermediate_output.open(encoding='utf-8') as new_tasks_file, \
                 self.yin_path(DUTY_CONFIG_PATH).open(encoding='utf-8') as duty_config_file:
                new_tasks = yaml.safe_load(new_tasks_file)
                duty_config = yaml.safe_load(duty_config_file)

            merge_duty_config = duty_config.get(CHANNEL_NAME)
            merge_duty_config['dated_tasks'] = new_tasks
            self.yin_path(DUTY_CONFIG_PATH).write_text(
                six.ensure_text(yaml.safe_dump(
                    data=merge_duty_config, default_flow_style=False, allow_unicode=True
                )),
                encoding='utf-8',
            )
        except Exception as e:
            logging.exception(e)
            self.set_info('Failed to write changes to file')
            raise

        try:
            self.git.add(str(self.yin_path(DUTY_CONFIG_PATH)))
            self.git.commit('-m', 'Write duty tasks for merge {} ({})'.format(
                self.Parameters.chromium_version, self.Context.merge_length_str
            ))

            with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                self.git.push('origin', '{}:refs/heads/{}'.format(pr_branch, pr_branch))
            pr = self.bb.create_pr('stardust', 'yin',
                                   'Write duty tasks for merge {} ({})'.format(
                                       self.Parameters.chromium_version, self.Context.merge_length_str
                                   ),
                                   description='Created from sandbox task {}'.format(get_task_link(self.id)),
                                   from_ref=pr_branch, to_ref=self.Parameters.branch,
                                   reviewers=self.Parameters.reviewers)
            self.set_info('Created <a href="{}">pull request</a>'.format(pr.web_url),
                          do_escape=False)
        except Exception as e:
            logging.exception(e)
            self.set_info('Failed to commit or push changes')
            raise
        finally:
            self.yin_path('diff.txt').write_text(
                six.ensure_text(self.git.diff('HEAD~')),
                encoding='utf-8',
            )
            diff_resource = DutyConfigDiff(self, 'Diff of config changes', str(self.yin_path('diff.txt')))
            sdk2.ResourceData(diff_resource).ready()

    def on_execute_impl(self, exit_stack):
        if self.Parameters.use_custom_template:
            # merging a template with itself yields it's content
            self.Context.merge_start_template = str(self.yin_path(self.Parameters.custom_template_path))
            self.Context.merge_end_template = str(self.yin_path(self.Parameters.custom_template_path))
        else:
            self.Context.merge_start_template = str(self.yin_path(COMMON_TEMPLATE_PATH))
            if int(self.Parameters.chromium_version) % 2:
                self.Context.merge_end_template = str(self.yin_path(FOUR_WEEKS_TEMPLATE_PATH))
                self.Context.merge_length_str = 'four-week merge'
            else:
                self.Context.merge_end_template = str(self.yin_path(THREE_WEEKS_TEMPLATE_PATH))
                self.Context.merge_length_str = 'three-week merge'

        super(BrowserMergeGenerateDutyTasks, self).on_execute_impl(exit_stack)
        with self.step('Update duty config', 'update_config'):
            self.write_tasks_to_config()
