import contextlib
import os

from sandbox import sdk2
from sandbox.common.types import misc
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.common import teamcity
from sandbox.sandboxsdk.environments import PipEnvironment, NodeJS


class BrowserMergeNodejsTask(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 10 * 1024  # 10GB
        dns = misc.DnsType.DNS64
        environments = [
            PipEnvironment('teamcity-client==4.0.0'),
            NodeJS('10.14.2'),
        ] + RunFromYinTask.Requirements.environments.default

    class Parameters(RunFromYinTask.Parameters):
        teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)
        serve_from_teamcity = sdk2.parameters.Bool('Use TC for links to artifacts', default=False)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff(
                'Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String(
                'Vault item with robot token',
                default='robot-bro-merge_token')

    def artifacts_path(self, *args):
        return self.path('artifacts', *args)

    @property
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=sdk2.Vault.data(self.Parameters.robot_token)
        )

    @contextlib.contextmanager
    def publish_artifacts(self):
        if not os.path.exists(str(self.artifacts_path())):
            os.makedirs(str(self.artifacts_path()))
        if not hasattr(self, 'artifacts_resource'):
            self.artifacts_resource = teamcity.TeamcityArtifacts(
                self, 'Teamcity artifacts', self.artifacts_path())
            publish = True
        else:
            publish = False

        try:
            if self.Parameters.serve_from_teamcity:
                build_type = self.teamcity_client.builds[self.Parameters.teamcity_build_id].build_type.id
                url = '{tc_url}repository/download/{build_type}/{build_id}:id'.format(
                    tc_url=self.teamcity_client.server_url,
                    build_type=build_type,
                    build_id=self.Parameters.teamcity_build_id,
                )
            else:
                url = self.artifacts_resource.http_proxy
            yield url
        finally:
            if publish:
                if not os.listdir(str(self.artifacts_path())):
                    # Touch a dummy file to prevent from resource break.
                    with open(str(self.artifacts_path('dummy')), 'w'):
                        pass

                sdk2.ResourceData(self.artifacts_resource).ready()

    def run_yin_script(self, script, args, extra_env=None, **kwargs):
        args = list(args) + [
            '-vvv',
            '--oauth-no-save',
            '--oauth-login', self.Parameters.robot_login,
            '--oauth-token-env', 'ROBOT_TOKEN',
        ]

        extra_env = extra_env or {}
        extra_env.update(
            ROBOT_TOKEN=sdk2.Vault.data(self.Parameters.robot_token))

        with self.publish_artifacts():
            super(BrowserMergeNodejsTask, self).run_yin_script(
                script, args, extra_env=extra_env, **kwargs)
