# coding=utf-8
import datetime
import logging
import os
import re
import time

import yaml

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.browser.merge.BrowserMergeBase import BrowserMergeBase
from sandbox.sandboxsdk.environments import PipEnvironment

MERGESTATES_FOLDER_PATH = 'src/build/yandex/merge/state'
PLATFORMS = ['desktop', 'android', 'ios']
DEFAULT_INFRA_REVIEWER = 'anata-m'
STATE_PR_REGEX = re.compile(r'^Enable (?P<state>\S+?) merge state$')
TIMELINE_PATH = 'src/build/yandex/merge/merge_timeline.yaml'


class BrowserMergeTurnStatesOn(BrowserMergeBase):
    class Parameters(BrowserMergeBase.Parameters):
        with sdk2.parameters.RadioGroup("Merge states source") as source:
            source.values.read_from_timeline = source.Value("Read from timeline", default=True)
            source.values.set_manually = source.Value("Set manually")
        with source.value['set_manually']:
            merge_states = sdk2.parameters.List('Merge states', required=True)
        branch = sdk2.parameters.String('Branch', required=True)
        platform_reviewers = sdk2.parameters.Dict(
            'Platform reviewers', required=False, default={platform: '' for platform in PLATFORMS})

    class Requirements(BrowserMergeBase.Requirements):
        environments = BrowserMergeBase.Requirements.environments.default + [
            PipEnvironment('yabrowser-blacklist-rt==0.0.10'),
        ]

    def turn_state_on(self, merge_state):
        import blacklist_rt

        self.git_helper.fetch_and_checkout(self.Parameters.branch)
        pr_branch = 'wp/robots/turn_states_on/{}'.format(int(time.time() * 1000))
        self.git.checkout('-f', '-B', pr_branch, self.Parameters.branch)
        platform, state = merge_state.split('/', 1)
        platform_reviewer = self.Parameters.platform_reviewers.get(platform)
        reviewers = [self.Parameters.platform_reviewers.get('infra', DEFAULT_INFRA_REVIEWER)]
        reviewers += [platform_reviewer] if platform_reviewer else []
        states_file_relative_path = os.path.join(MERGESTATES_FOLDER_PATH, '{}.yaml'.format(platform))
        states_file_path = self.repo_path(states_file_relative_path)
        with open(states_file_path, 'r') as f:
            states, formatting = blacklist_rt.load(f)
        states[self.Parameters.branch][state] = 'yes'
        with open(states_file_path, 'w') as f:
            blacklist_rt.dump(f, states, formatting)
        self.git.add(states_file_relative_path)
        self.git.commit('-m', 'Enable {} merge state'.format(merge_state))
        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            self.git.push('origin', '{}:refs/heads/{}'.format(pr_branch, pr_branch))
        pr = self.bb.create_pr('stardust', 'browser', 'Enable {} merge state'.format(merge_state), description='',
                               from_ref=pr_branch, to_ref=self.Parameters.branch, reviewers=reviewers)
        self.set_info('Created <a href="{}">pull request</a> for state {}'.format(pr.web_url, merge_state),
                      do_escape=False)

    def get_disabled_states(self):
        import blacklist_rt

        disabled_states = set()
        for platform in PLATFORMS:
            states_file_relative_path = os.path.join(MERGESTATES_FOLDER_PATH, '{}.yaml'.format(platform))
            states_file_path = self.repo_path(states_file_relative_path)
            with open(states_file_path, 'r') as f:
                states, _ = blacklist_rt.load(f)
            for state, status in states[self.Parameters.branch].iteritems():
                if not status:
                    disabled_states.add('{}/{}'.format(platform, state))
        logging.info("Got disabled states: %s", disabled_states)
        return disabled_states

    def get_states_that_should_be_enabled_by_timeline(self):
        timeline_file_path = self.repo_path(TIMELINE_PATH)
        with open(timeline_file_path, 'r') as f:
            timeline = yaml.safe_load(f)
        tomorrow = datetime.date.today() + datetime.timedelta(days=1)
        states_whose_time_has_come = set()
        for date_str, states in timeline.iteritems():
            date = datetime.datetime.strptime(date_str, '%Y-%m-%d').date()
            if states and date <= tomorrow:
                for state in states:
                    for branch, actual_state in state.iteritems():
                        if branch == self.Parameters.branch:
                            states_whose_time_has_come.add(actual_state)
        logging.info("Got states that should be enabled: %s", states_whose_time_has_come)
        return states_whose_time_has_come

    def get_states_with_prs(self):
        states_with_prs = set()
        for pr in self.bb.pull_requests('stardust', 'browser', self.Parameters.branch):
            match = STATE_PR_REGEX.match(pr.title)
            if match:
                states_with_prs.add(match.group('state'))
        logging.info("Got states with existing prs: %s", states_with_prs)
        return states_with_prs

    def get_states_to_enable(self):
        states_to_enable = self.get_disabled_states().intersection(self.get_states_that_should_be_enabled_by_timeline())
        return states_to_enable - self.get_states_with_prs()

    def on_execute(self):
        self.clone_repo(self.Parameters.branch)
        failures = False
        states = self.Parameters.merge_states or self.get_states_to_enable()
        logging.info("Got states to enable: %s", states)
        for state in states:
            try:
                self.turn_state_on(state)
            except Exception as e:
                logging.exception(e)
                failures = True
                self.set_info('Failed to create pull request for state {}: {}'.format(state, e.message))
        if failures:
            raise TaskFailure('There were errors during task')
