import logging

import os

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm
from sandbox.common.utils import singleton_property

from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.common.bitbucket import (
    BitBucket, DEFAULT_BITBUCKET_URL, TESTING_BITBUCKET_URL)
from sandbox.projects.browser.common.git.git_cli import GitCli


class BrowserMirrorGitRepository(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 30 * 1024  # 30GB
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024
        environments = (
            GitEnvironment('2.19'),
        )
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        upstream_url = sdk2.parameters.String(
            'Upstream URL', required=True)
        use_mirror = sdk2.parameters.Bool(
            'Use git mirror.', default=False)
        refs = sdk2.parameters.List('Refs to fetch', default=['refs/heads/*', 'refs/tags/*'])
        with sdk2.parameters.Group('BitBucket settings'):
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
            bitbucket_project = sdk2.parameters.String(
                'Project', default='CHROMIUM')
            bitbucket_repo = sdk2.parameters.String('Repository')

        with sdk2.parameters.Group("Credentials"):
            robot_login = sdk2.parameters.String(
                "Login for teamcity & bitbucket", default="robot-bro-merge")
            robot_password_vault = sdk2.parameters.String(
                "Vault item with password for teamcity & bitbucket",
                default="robot-bro-merge_password")
            robot_ssh_key_vault = sdk2.parameters.String(
                "Vault item with ssh key for bitbucket",
                default="robot-bro-merge_ssh_key")

    @property
    def repo_path(self):
        return str(self.path('mirror'))

    @singleton_property
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = TESTING_BITBUCKET_URL
        else:
            bitbucket_url = DEFAULT_BITBUCKET_URL
        return BitBucket(bitbucket_url, self.Parameters.robot_login,
                         sdk2.Vault.data(self.Parameters.robot_password_vault))

    @singleton_property
    def git(self):
        return GitCli(self.repo_path)

    def on_execute(self):
        upstream_url = self.Parameters.upstream_url

        if self.Parameters.use_mirror:
            vcs_root = repositories.bitbucket_vcs_root(
                self.Parameters.bitbucket_project,
                self.Parameters.bitbucket_repo,
                testing=self.Parameters.use_test_bitbucket)
            vcs_root.update_cache_repo()
            vcs_root.clone_bare(self.repo_path)
        else:
            clone_url = self.bb.get_repo_clone_url(
                self.Parameters.bitbucket_project,
                self.Parameters.bitbucket_repo)
            os.mkdir(self.repo_path)
            self.git.init('--bare', self.repo_path)
            self.git.remote('add', 'origin', clone_url)

        self.git.remote('add', 'upstream', upstream_url)
        logging.info('Fetching repository from %s', upstream_url)
        refs = [
            '{0}:{0}'.format(ref) for ref in self.Parameters.refs
        ]
        with sdk2.ssh.Key(
                self, self.Parameters.robot_ssh_key_vault, None):
            self.git.fetch('upstream', *refs)
            self.git.push('origin', '-f', *refs)

        self.update_repo_label()

    def update_repo_label(self):
        if not self.scheduler:
            return

        scheduler_label = 'sandbox-scheduler-{}'.format(
            self.scheduler)

        for label in self.bb.get_repository_labels(
            self.Parameters.bitbucket_project,
            self.Parameters.bitbucket_repo,
        ):
            if label['name'] == scheduler_label:
                return

        self.bb.create_repository_label(
            self.Parameters.bitbucket_project,
            self.Parameters.bitbucket_repo,
            scheduler_label)
