import logging
from collections import defaultdict, Counter

from sandbox import sdk2

BUILDTYPE_PLATFORM = {
    'Browser_Tests_UnitTestsLinuxGn': 'linux',
    'Browser_Tests_UnitTestsLinuxStatic': 'linux.static',
    'Browser_Tests_UnitTestsWinGn': 'win',
    'Browser_Tests_UnitTestsWin64': 'win64',
    'Browser_Tests_UnitTestsWinStatic': 'win.static',
    'Browser_Tests_UnitTestsMac': 'mac',
    'Browser_Tests_UnitTestsMacStatic': 'mac.static',
    'Browser_Tests_UnitTestsAndroid': 'android',
    'Browser_Tests_UnitTestsIos': 'ios',
}

PLATFORM_CONFIG = {
    'linux': 'src/build/yandex/ci/configs/platforms/linux.yaml',
    'linux.static': 'src/build/yandex/ci/configs/static/linux.yaml',
    'win': 'src/build/yandex/ci/configs/platforms/win.yaml',
    'win64': 'src/build/yandex/ci/configs/test_flavors/win.x64.yaml',
    'win.static': 'src/build/yandex/ci/configs/static/win.yaml',
    'mac': 'src/build/yandex/ci/configs/platforms/mac.yaml',
    'mac.static': 'src/build/yandex/ci/configs/static/mac.yaml',
    'android': 'src/build/yandex/ci/configs/platforms/android.yaml',
    'ios': 'src/build/yandex/ci/configs/platforms/ios.yaml',
}


def get_broken_tests(build_ids, teamcity_client, build_counts, broken_test_threshold,
                     disable_other_platforms_threshold=100):
    build_type_test_stats = defaultdict(Counter)  # build_type -> test_name -> failure count
    thresholds = {
        build_type: int(counts) * broken_test_threshold / 100
        for build_type, counts in build_counts.iteritems()
    }
    disable_others_thresholds = {
        build_type: int(counts) * disable_other_platforms_threshold / 100
        for build_type, counts in build_counts.iteritems()
    }

    for build_id in build_ids:
        build_type_test_stats[teamcity_client.builds[build_id].data.buildTypeId].update(
            test['name'].split(': ', 1)[-1]  # teamcity reports some tests as "suite_name: full_test_name"
            for test in teamcity_client.rest_api.testOccurrences.get(locator=dict(
                build=dict(id=build_id), count=100000, status='FAILURE')).get('testOccurrence', [])
        )

    broken_tests = {}
    for build_type, stats in build_type_test_stats.iteritems():
        broken_tests[BUILDTYPE_PLATFORM[build_type]] = set(
            test_name
            for test_name, failure_count in stats.iteritems()
            if failure_count > thresholds[build_type] or (
                # checking if this test failed on some other platform
                failure_count > disable_others_thresholds[build_type] and
                any(st[test_name] > thresholds[bt]
                    for bt, st in build_type_test_stats.iteritems()
                    if bt != build_type)
            )
        )

    return broken_tests


def group_tests(broken_tests):
    result = dict()

    result['linux'] = broken_tests.get('linux', set())
    result['linux.static'] = broken_tests.get('linux.static', set())
    result['win'] = broken_tests.get('win', set())
    result['win.static'] = broken_tests.get('win.static', set())
    result['mac'] = broken_tests.get('mac', set())
    result['mac.static'] = broken_tests.get('mac.static', set())
    result['android'] = broken_tests.get('android', set())
    result['ios'] = broken_tests.get('ios', set())
    result['win64'] = broken_tests.get('win64', set()) - result['win']

    return result


def check_merge_in_progress(bb_client, branches):
    for branch in branches:
        if not bb_client.compare_commits('STARDUST', 'browser', branch, 'master-next'):
            logging.info('Branch %s already merged into master-next.', branch)
            return False
    return True


def get_branch_lock(bb_client, branch, project='stardust', repo='browser'):
    # suppose that branch is "locked" if there is a READ_ONLY restriction for branch
    branch_restrictions = bb_client.get_restrictions(
        project, repo,
        restriction_type='read-only',
        matcher_type='BRANCH',
        matcher_id='refs/heads/' + branch
    )
    logging.debug('branch %s has this lock: %s', branch, branch_restrictions)
    if branch_restrictions:
        return branch_restrictions[0]
    else:
        return None


class BlacklistsDiff(sdk2.Resource):
    pass
