import datetime
import textwrap
import logging

from sandbox.projects.browser.merge.common import python_seriallization

# Dependencies with these words will not be fetched and will not be saved
# to snapshot dependencies.
DEPS_BLACKLIST = [
    'octane',
    'reference_build',
    'gsutil',
    'third_party/skia/common',
]

# Sometime chromium DEPS requires different revisions of same project for
# same dependency. In a usual way we will not commit these dependencies into
# snapshot and save them to the snapshot dependencies. Deps from this list
# will be commited into snapshot.
DEPS_IGNORE_CONFLICTS = [
    'src/breakpad/src',
    'src/tools/gyp',
]

# Dependencies with these words will not be commited to snapshot and will
# be saved into snapshot dependencies. Usually it means that dependency is
# required only for one platform and too heavy to commit it into snapshot.
DEPS_PRESERVE = [
    'android_tools',
    'android_ndk',
    'depot_tools',

    # ios material design repos
    'material_components_ios',
    'material_font_disk_loader_ios',
    'material_internationalization_ios',
    'material_roboto_font_loader_ios',
    'material_sprited_animation_view_ios',
    'material_text_accessibility_ios',
    'motion_interchange_objc',
    'motion_animator_objc',
    'motion_transitioning_objc',
]

# We have some dependencies which repository does not match pattern
# browser-deps-* for historical reasons. Specify repository names for them
# here.
DEPS_REDIRECT = {
    'src/third_party/depot_tools': 'depot_tools',
    'src/third_party/android_ndk':
        'browser-deps-src-third_party-android_tools-ndk',
    'src/third_party/angle/third_party/android_ndk':
        'browser-deps-src-third_party-android_tools-ndk',
    'src/third_party/angle/third_party/depot_tools':
        'depot_tools',
}


class Deps(object):
    '''
    Chromium DEPS file.
    '''

    def __init__(self, deps_dict):
        self.deps_dict = deps_dict

    @classmethod
    def from_string(cls, content):
        '''
        Construct Deps object with file contents.
        '''

        # Evaluates gclient deps file as a python.
        # Replaces Var('VARNAME') calls with '{VARNAME}' then uses
        # str.format(**vars) to expand variables.
        # Serializes Str('value') calls.
        def exec_deps(content):
            global_vars = {
                'Var': lambda name: '{{{name}}}'.format(name=name),
                'Str': lambda *args, **kwargs: python_seriallization.NamedCall('Str', args, kwargs),
            }
            local_vars = {}
            exec content in global_vars, local_vars
            return local_vars

        local_vars = exec_deps(content)

        return cls(local_vars).merge_deps_os_to_deps()

    @property
    def deps(self):
        return self.deps_dict.setdefault('deps', {})

    @property
    def deps_os(self):
        return self.deps_dict.get('deps_os', {})

    @property
    def all_paths_set(self):
        '''
        Set of all deps/deps_os paths.
        '''

        return set(self.deps.keys())

    @property
    def cipd_deps(self):
        '''
        Set of cipd deps.
        '''
        return {
            path
            for path, dep in self.deps.iteritems()
            if isinstance(dep, dict) and dep.get('dep_type') == 'cipd'
        }

    @property
    def blacklisted(self):
        '''
        Set of paths, blacklisted by DEPS_BLACKLIST
        '''

        return {
            path
            for path in self.all_paths_set
            if any(needle in path for needle in DEPS_BLACKLIST)
        }

    @property
    def preserved(self):
        '''
        Set of paths, matched by DEPS_PRESERVE and cipd_deps.
        '''

        return {
            dep
            for dep, dep_url in self.deps.iteritems()
            if any(needle in dep for needle in DEPS_PRESERVE)
        } | self.cipd_deps

    @property
    def excluded(self):
        '''
        List of deps which must not be commited into snapshot.
        '''

        return sorted(self.blacklisted |
                      self.preserved)

    def merge_deps_os_to_deps(self):
        extra_deps = {}
        OS_CONDITIONS = {
            'unix': 'checkout_linux',
            'android': 'checkout_andoroid',
            'win': 'checkout_win',
            'mac': 'checkout_mac',
            'ios': 'checkout_ios',
            'fuchsia': 'checkout_fuchsia',
            'chromeos': 'checkout_chromeos',
        }
        for os, os_deps in self.deps_os.iteritems():
            os_condition = OS_CONDITIONS[os]
            for path, dep in os_deps.iteritems():
                if not isinstance(dep, dict):
                    dep = {'dep_type': 'git', 'url': dep['url']}
                else:
                    dep = dict(dep)
                dep_condition = dep.get('condition')
                if dep_condition:
                    del dep['condition']
                    dep_condition = '({} and ({}))'.format(
                        os_condition, dep_condition)
                dep = extra_deps.setdefault(path, dep)
                if 'condition' in dep:
                    if dep['condition'][:-1] != ')':
                        dep['condition'] = '({})'.format(dep['condition'])
                    dep['condition'] += ' or ' + dep_condition
                else:
                    dep['condition'] = dep_condition

        extra_hooks = []
        for os, os_hooks in self.hooks_os.iteritems():
            os_condition = OS_CONDITIONS[os]
            for hook in os_hooks:
                hook = dict(hook)
                hook_condition = hook.get('condition')
                if hook_condition:
                    hook_condition = '{} and ({})'.format(
                        os_condition, hook_condition)
                else:
                    hook_condition = os_condition
                hook['condition'] = hook_condition
                extra_hooks.append(hook)

        extra_attrs = {
            attr_name: self.deps_dict[attr_name]
            for attr_name in (
                'gclient_gn_args_file',
                'gclient_gn_args',
                'recursedeps',
            ) if attr_name in self.deps_dict
        }

        return Deps(dict(
            vars=dict(self.vars),
            deps=dict(self.deps, **extra_deps),
            hooks=self.hooks + extra_hooks,
            **extra_attrs
        ))

    def filter_out_cipd(self):
        extra_attrs = {
            attr_name: self.deps_dict[attr_name]
            for attr_name in (
                'gclient_gn_args_file',
                'gclient_gn_args',
                'recursedeps',
            ) if attr_name in self.deps_dict
        }

        return Deps(dict(
            vars=dict(self.vars),
            deps={
                path: dep
                for path, dep in self.deps.iteritems()
                if isinstance(dep, str) or dep.get('dep_type', 'git') != 'cipd'
            },
            hooks=list(self.hooks),
            **extra_attrs
        ))

    def filter_out_excluded(self):
        '''
        Return Deps which should be used to create snapshot.
        '''
        excluded = self.excluded
        filtered_deps = {}
        for path, dep in self.deps.iteritems():
            if path not in excluded:
                filtered_deps[path] = dep

        extra_attrs = {
            attr_name: self.deps_dict[attr_name]
            for attr_name in (
                'recursedeps',
            ) if attr_name in self.deps_dict
        }

        return Deps(dict(
            vars=self.vars,
            deps=filtered_deps,
            hooks=self.hooks,
            **extra_attrs
        ))

    def filter_out_recursedeps(self):
        return Deps({
            k: v for k, v in self.deps_dict.iteritems()
            if k != 'recursedeps'
        })

    @property
    def hooks(self):
        return self.deps_dict.setdefault('hooks', [])

    @property
    def vars(self):
        return self.deps_dict.setdefault('vars', {})

    @property
    def recursedeps(self):
        return self.deps_dict.setdefault('recursedeps', [])

    @property
    def hooks_os(self):
        return self.deps_dict.get('hooks_os', {})

    def deps_to_preserve(self):
        '''
        Returns deps which must be written into snapshot dependencies.
        '''

        result_deps = {
            path: dict(dep)
            for path, dep in self.deps.iteritems()
            if path in self.preserved
        }

        extra_attrs = {
            attr_name: self.deps_dict[attr_name]
            for attr_name in (
                'gclient_gn_args_file',
                'gclient_gn_args',
                'recursedeps',
            ) if attr_name in self.deps_dict
        }

        return Deps(dict(
            vars=self.vars,
            deps=result_deps,
            hooks=self.hooks,
            **extra_attrs
        ))

    def mirror_url(self, bb, project, name, spec):
        '''
        Redirects dependency url to the bitbucket.

        Checks that required revision exists in the mirror.
        '''

        repo = 'browser-deps-' + name.replace('/', '-')
        if name in DEPS_REDIRECT:
            repo = DEPS_REDIRECT[name]

        if isinstance(spec, basestring):
            spec = {
                'url': spec,
                'dep_type': 'git',
            }

        # Do not redirect cipd deps.
        if spec.get('dep_type') == 'cipd':
            return spec

        assert '@' in spec['url'], (
            'No revision found in dependency URL: {}'.format(spec['url']))
        commit = spec['url'].rsplit('@', 1)[-1]

        mirror_url = bb.get_repo_clone_url(project, repo, name='http')
        assert bb.get_commit(project, repo, commit)['id'] == commit
        spec['url'] = '@'.join((mirror_url, commit))

        return spec

    def redirect_to_mirror(self, bb, project):
        '''
        Redirects each dependency to bitbucket mirror and returns deps.
        '''

        result_deps = {}
        for dep, url in self.deps.iteritems():
            result_deps[dep] = self.mirror_url(bb, project, dep, url)

        extra_attrs = {
            attr_name: self.deps_dict[attr_name]
            for attr_name in (
                'gclient_gn_args_file',
                'gclient_gn_args',
                'recursedeps',
            ) if attr_name in self.deps_dict
        }

        return Deps(dict(
            vars=self.vars,
            deps=result_deps,
            hooks=self.hooks,
            **extra_attrs
        ))

    def seriallize(self, year):
        result = textwrap.dedent('''\
            # Copyright (c) {year} Yandex LLC. All rights reserved.
            # Author: BROWSER_MAKE_CHROMIUM_SNAPSHOT task from sandbox.
            ''').format(year=year)

        result += python_seriallization.format_as_python(
            self.deps_dict, indent=' ' * 4, prevent_conflicts=True)
        return result

    def save(self, path):
        '''
        Saves dependencies into file.
        '''

        result = self.seriallize(datetime.date.today().year)

        logging.info('Saving deps:\n%s', result)

        with open(path, 'w') as f:
            f.write(result)

    def __eq__(self, other):
        return self.deps_dict == other.deps_dict
