import json
import os

from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn

from sandbox.projects.browser.common.RunBrowserScript import RunBrowserScript
from sandbox.projects.browser.common import LXC_RESOURCE_ID
from sandbox.projects.browser.merge.conflicts.BrowserAutoRevert import BrowserAutoRevert

from sandbox.sdk2.helpers import ProcessLog, subprocess

from sandbox import sdk2


class BrowserAutoFixit(RunBrowserScript):
    BROWSER_REPO_NAME = 'browser'
    BROWSER_PROJECT = 'STARDUST'
    TARGET_MODULE = 'yandex.sheriff.autofixit'
    secret_envvars = RunBrowserScript.secret_envvars + ('robot_token',)

    class Requirements(RunBrowserScript.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        cores = 16
        ram = 32 * 1024

        class Caches(RunBrowserScript.Requirements.Caches):
            pass

    class Parameters(RunBrowserScript.Parameters):
        commit = RunBrowserScript.Parameters.commit(ui=None)
        platform = RunBrowserScript.Parameters.platform(ui=None)
        suspend_after = RunBrowserScript.Parameters.suspend_after(ui=None)
        general_settings = RunBrowserScript.Parameters.general_settings(ui=None)
        debug_settings = RunBrowserScript.Parameters.debug_settings(ui=None)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.String(
                'Login for teamcity & bitbucket', default='robot-bro-merge')
            robot_token_vault = sdk2.parameters.String(
                'Vault item with token for bitbucket',
                default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key or bitbucket',
                default='robot-bro-merge_ssh_key')

        with sdk2.parameters.Group('Pull request and configuration params') as bb_group:
            limit = sdk2.parameters.Integer('Max number of files in a PR', default=100)
            configuration = sdk2.parameters.String('Build configuration (Release/Debug/Default etc)',
                                                   default='Default')

        _container = sdk2.parameters.Container(
            'Linux container', default_value=LXC_RESOURCE_ID)

        notifications = (
            sdk2.Notification(
                (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                 ctt.Status.SUCCESS, ctt.Status.STOPPED),
                ('anata-m',),
                ctn.Transport.EMAIL
            )
        )

    @property
    def script_out_path(self):
        return os.path.abspath('fixit_prs.json')

    def script_cmd(self, python_executable):
        cmd = [str(python_executable), '-m', self.TARGET_MODULE,
               str(self.browser_path()), self.Parameters.branch,
               self.script_out_path,
               '--bb-project', self.BROWSER_PROJECT,
               '--bb-repo', self.BROWSER_REPO_NAME,
               '--limit', str(self.Parameters.limit),
               '--configuration', self.Parameters.configuration,
               '--oauth-login', self.Parameters.robot_login,
               '--oauth-token-env', 'robot_token',
               '--oauth-no-save']
        if self.Parameters.use_test_bitbucket:
            cmd.append('--use-test-bitbucket')

        return cmd

    def script_extra_env(self):
        env = super(BrowserAutoFixit, self).script_extra_env()
        env.update({'robot_token': sdk2.Vault.data(self.Parameters.robot_token_vault)})
        return env

    def gclient_runhooks_extra_env(self):
        env = {
            'DC_CACHE_TYPE': 'none',
            'ENABLE_YAML_MERGER': '0',
            'INCREMENT_YANDEX_VERSION': '0',
        }
        return env

    def gclient_runhooks(self):
        extra_env = self.gclient_runhooks_extra_env()
        self.print_env(extra_env, 'GClient runhooks extra environment:')
        env = dict(os.environ, **extra_env)
        with ProcessLog(self, logger='browser_gclient_runhooks') as log:
            subprocess.check_call(
                ['gclient', 'runhooks', '--verbose', '--deps={}'.format(self.Parameters.platform)],
                cwd=str(self.browser_path()), stdout=log.stdout, stderr=log.stdout, env=env)

    def run_script(self, python_executable):
        self.gclient_runhooks()
        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            super(BrowserAutoFixit, self).run_script(python_executable)

    def on_execute_impl(self, exit_stack):
        exit_stack.callback(self.on_execute_finally)
        super(BrowserAutoFixit, self).on_execute_impl(exit_stack)

    def on_execute_finally(self):
        if os.path.exists(self.script_out_path):
            with open(self.script_out_path) as f:
                prs = json.load(f)
        else:
            prs = []
        if prs:
            revert_task = BrowserAutoRevert(
                self, owner=self.Parameters.owner,
                description='Created from {}'.format(self.id),
                prs=prs,
                branch=self.Parameters.branch,
                use_test_bitbucket=self.Parameters.use_test_bitbucket,
                bb_project=self.BROWSER_PROJECT,
                browser_repo=self.BROWSER_REPO_NAME,
                robot_login=self.Parameters.robot_login,
                robot_token_vault=self.Parameters.robot_token_vault,
                robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault,
                notifications=self.Parameters.notifications,
            )
            self.Context.revert_task = revert_task.id
            revert_task.enqueue()
        else:
            self.set_info('WARN: Autofixit created no prs')
