import json
import os

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn

from sandbox.projects.browser.merge.conflicts.BrowserAutoRevert import BrowserAutoRevert
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask

DEFAULT_PROJECT = 'STARDUST'
DEFAULT_BROWSER_REPO = 'browser'


class BrowserAutoResolve(RunFromYinTask):
    class Parameters(RunFromYinTask.Parameters):
        branch = sdk2.parameters.String("branch", required=True)
        limit = sdk2.parameters.Integer("Max number of files in a PR", default=100)
        additional_reviewer = sdk2.parameters.Staff("Additional reviewer", default=None)

        notifications = (
            sdk2.Notification(
                (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                 ctt.Status.SUCCESS, ctt.Status.STOPPED),
                ('myxomopla', 'hpotter'),
                ctn.Transport.EMAIL
            )
        )

        with sdk2.parameters.Group('BitBucket settings') as bitbucket_group:
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
            bb_project = sdk2.parameters.String(
                'Project', default=DEFAULT_PROJECT)
            browser_repo = sdk2.parameters.String(
                'Browser repository', default=DEFAULT_BROWSER_REPO)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.String(
                'Login for teamcity & bitbucket', default='robot-bro-merge')
            robot_token_vault = sdk2.parameters.String(
                'Vault item with token for bitbucket',
                default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key for bitbucket',
                default='robot-bro-merge_ssh_key')

    @property
    def repo_path(self):
        return str(self.path('browser'))

    def on_execute(self):
        vcs_root = repositories.bitbucket_vcs_root(
            self.Parameters.bb_project, self.Parameters.browser_repo,
            testing=self.Parameters.use_test_bitbucket)
        # FIXME update_cache is used here, because without it cloning branch fails on branch, which is a suffix of
        # another branch. e.g. cloning feature/19.10 fails if branch bundle/feature/19.10 exists.
        vcs_root.update_cache_repo()
        vcs_root.clone(self.repo_path, self.Parameters.branch)

        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            script_res_path = os.path.abspath('resolved_prs.json')
            script_args = [self.repo_path, self.Parameters.branch,
                           script_res_path,
                           '--bb-project', self.Parameters.bb_project,
                           '--bb-repo', self.Parameters.browser_repo,
                           '--limit', str(self.Parameters.limit),
                           '--oauth-login', self.Parameters.robot_login,
                           '--oauth-token-env', 'robot_token',
                           '--oauth-no-save']
            if self.Parameters.additional_reviewer:
                script_args += ['--additional-reviewer', self.Parameters.additional_reviewer]
            self.run_yin_script('yin.merge.conflicts.autoresolve',
                                script_args,
                                extra_env=dict(
                                    robot_token=sdk2.Vault.data(
                                        self.Parameters.robot_token_vault)))
        with open(script_res_path) as f:
            prs = json.load(f)
        if prs:
            revert_task = BrowserAutoRevert(
                self, owner=self.Parameters.owner,
                description='Created from {}'.format(self.id),
                prs=prs,
                branch=self.Parameters.branch,
                use_test_bitbucket=self.Parameters.use_test_bitbucket,
                bb_project=self.Parameters.bb_project,
                browser_repo=self.Parameters.browser_repo,
                robot_login=self.Parameters.robot_login,
                robot_token_vault=self.Parameters.robot_token_vault,
                robot_ssh_key_vault=self.Parameters.robot_ssh_key_vault,
                notifications=self.Parameters.notifications,
            )
            self.Context.revert_task = revert_task.id
            revert_task.enqueue()
        else:
            self.set_info("WARN: Autoresolve created no prs")
