import sandbox.common.types.client as ctc

from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.merge.conflicts.BrowserAutoRevert.revert import do_reverts

from sandbox import sdk2

DEFAULT_PROJECT = 'STARDUST'
DEFAULT_BROWSER_REPO = 'browser'


class BrowserAutoRevert(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.Group.LINUX
        environments = (
            GitEnvironment('2.19'),
        )
        cores = 16
        ram = 32 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String('Branch to clone', default='master-next')
        prs = sdk2.parameters.List('Pr ids to autorevert', required=True)
        wait_time = sdk2.parameters.Integer(
            'Minutes to wait between autoreverts',
            default=15)

        with sdk2.parameters.Group('BitBucket settings') as bitbucket_group:
            use_test_bitbucket = sdk2.parameters.Bool('Use test BitBucket')
            bb_project = sdk2.parameters.String(
                'Project', default=DEFAULT_PROJECT)
            bb_repo = sdk2.parameters.String(
                'Browser repository', default=DEFAULT_BROWSER_REPO)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.String(
                'Login for teamcity & bitbucket', default='robot-bro-merge')
            robot_token_vault = sdk2.parameters.String(
                'Vault item with token for bitbucket',
                default='robot-bro-merge_token')
            robot_ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key for bitbucket',
                default='robot-bro-merge_ssh_key')

    @property
    def repo_path(self):
        return str(self.path('browser'))

    @property
    def bb(self):
        if self.Parameters.use_test_bitbucket:
            bitbucket_url = bitbucket.TESTING_BITBUCKET_URL
        else:
            bitbucket_url = bitbucket.DEFAULT_BITBUCKET_URL
        return bitbucket.BitBucket(
            bitbucket_url,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.robot_token_vault))

    def on_save(self):
        self.Context.prs_left = self.Parameters.prs

    def on_execute(self):
        vcs_root = repositories.bitbucket_vcs_root(
            self.Parameters.bb_project, self.Parameters.bb_repo,
            testing=self.Parameters.use_test_bitbucket)
        # FIXME update_cache is used here, because without it cloning branch fails on branch, which is a suffix of
        # another branch. e.g. cloning feature/19.10 fails if branch bundle/feature/19.10 exists.
        vcs_root.update_cache_repo()
        vcs_root.clone(self.repo_path, self.Parameters.branch)

        with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
            self.Context.prs_left = do_reverts(self.repo_path,
                                               self.Parameters.bb_project, self.Parameters.bb_repo,
                                               self.Context.prs_left, self.bb)
        if self.Context.prs_left:
            raise sdk2.WaitTime(self.Parameters.wait_time * 60)
