import logging
import re

from sandbox.projects.browser.common import log_calls
from sandbox.projects.browser.common.git.git_cli import GitHelper

MERGE_BRANCH_REGEX = re.compile(r'merge-\d+\.\d+\.\d+/\d+')
BRANCH_PREFIX = 'wp/'
REVERT_COMMENT_TEXT = 'Revert'
GIT_USER = 'robot-bro-merge'
GIT_USER_EMAIL = 'roboto-bro-merge@yandex-team.ru'


class AutoRevert(object):

    def __init__(self, repo_path, project, repo, bitbucket_client):
        self.browser_git_repo = GitHelper(repo_path, config={
            'user.name': GIT_USER, 'user.email': GIT_USER_EMAIL})
        self.project = project
        self.repo = repo
        self.stash = bitbucket_client

    def get_open_prs(self, ids):
        ids = {int(pr_id) for pr_id in ids}
        return [pr for pr in
                self.stash.pull_requests(self.project, self.repo)
                if pr.id in ids]

    def get_pr_changes(self, pull_request):
        # type: (PullRequestDetails) -> dict
        return self.stash.pull_request_changes(self.project, self.repo,
                                               pull_request.id)

    def get_revert_comment_activities(self, pull_request):
        # type: (PullRequestDetails) -> list
        activities = self.stash.pull_request_activities(
            self.project, self.repo, pull_request.id)
        return [activity for activity in activities
                if activity['action'] == 'COMMENTED' and
                REVERT_COMMENT_TEXT in activity['comment']['text']]

    @staticmethod
    def get_comment_file(activity):
        return activity.get('commentAnchor', {}).get('path')

    @staticmethod
    def get_comment_id(activity):
        return activity['comment']['id']

    def get_revert_info_for_pr(self, pull_request):
        """ Returns a list of tuples with files, 'Revert' comments and pr_id"""
        changed_files = self.get_pr_changes(pull_request)
        revert_activities = self.get_revert_comment_activities(pull_request)
        return [
            (self.get_comment_file(comment), comment, pull_request.id)
            for comment in revert_activities
            if self.get_comment_file(comment) in changed_files
        ]

    def get_revert_info(self, auto_resolve_pull_requests):
        # type: (list(PullRequestDetails)) -> dict
        """ Returns dict(branch name => files & comments to revert). """
        reverts = {}
        for pull_request in auto_resolve_pull_requests:
            branch = pull_request.fromRef['displayId']
            revert_info = self.get_revert_info_for_pr(pull_request)
            if revert_info:
                reverts.setdefault(branch, []).extend(revert_info)
        logging.info('Reverts: {}'.format(reverts))
        return reverts

    @log_calls(True, 'branch: {branch}, revert_file: {revert_file}')
    def get_automated_commit(self, branch, revert_file):
        return self.browser_git_repo.commit_by_message(
            r'.*\b{file}$'.format(file=revert_file),
            branch=branch)

    @log_calls(True, 'hash: {hash}, branch: {branch}')
    def remove_commit(self, hash, branch):
        self.browser_git_repo.git.rebase(
            '--onto', '{hash}^'.format(hash=hash), hash, branch)

    @log_calls(True, 'branch: {branch}')
    def push_changes(self, branch):
        self.browser_git_repo.git.push('-f', 'origin', branch)

    @log_calls(True)
    def resolve(self, reverts):
        for branch in reverts:
            self.browser_git_repo.fetch_and_checkout(branch)
            pr_id = None
            reverted_files = set()
            for revert_file, _, pull_request in reverts[branch]:
                if revert_file in reverted_files:
                    continue
                reverted_files.add(revert_file)
                commit = self.get_automated_commit(branch, revert_file)
                logging.info('Reverted commit: {}'.format(commit))
                self.remove_commit(commit, branch)
                # We assume there's only one PR for every branch.
                pr_id = pull_request
            if pr_id is not None:
                self.push_changes(branch)
                self.stash.press_run_builds(self.project, self.repo, pr_id)

    @log_calls(True)
    def reply_comments(self, reverts):
        for branch_reverts in reverts.itervalues():
            for _, comment, pull_request_id in branch_reverts:
                comment_id = self.get_comment_id(comment)
                logging.info('Reply to PR#{pr_id} comment#{comment_id}'.format(
                    pr_id=pull_request_id, comment_id=comment_id))
                self.stash.comment_pull_request(
                    self.project, self.repo, pull_request_id,
                    'Done', comment_id)


def do_reverts(repo_path, project, repo, pull_requests, bibucket_client):
    auto_reverter = AutoRevert(repo_path, project, repo, bibucket_client)
    auto_resolve_pull_requests = auto_reverter.get_open_prs(pull_requests)
    # save ids of prs that are still to be monitored after reverting
    ids = [pr.id for pr in auto_resolve_pull_requests]
    if len(auto_resolve_pull_requests):
        reverts = auto_reverter.get_revert_info(auto_resolve_pull_requests)
        if reverts:
            auto_reverter.resolve(reverts)
            auto_reverter.reply_comments(reverts)
    return ids
