import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.merge.grupper import (
    BrowserMergeScatterTests, BrowserMergeYamlToIssues
)


class BrowserMergeGroupBlockers(RunFromYinTask):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 * 1024  # 1GB
        dns = misc.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        yin_branch = sdk2.parameters.String('Branch in YIN repo to get scripts from',
                                            default='master')
        merge_tests_branch = sdk2.parameters.String(
            'Scatter and group tests from this branch', required=True,
        )
        merge_version = sdk2.parameters.String('Merge version, with patch, e.g. 83.0.4103.5', required=True)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff('Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String('Vault item with robot token', default='robot-bro-merge_token')

        with sdk2.parameters.Output:
            grupper_addr = sdk2.parameters.String("IP address of grupper")

    class Context(sdk2.Task.Context):
        scatter_task = None
        grupper_task = None
        yaml2issues_task = None

    def on_execute(self):
        with self.memoize_stage.launch_scatter_tests():
            scatter_task = BrowserMergeScatterTests.BrowserMergeScatterTests(
                self,
                yin_branch=self.Parameters.yin_branch,
                tc_branch=self.Parameters.merge_tests_branch,
                checkout_branch=self.Parameters.merge_tests_branch,
                launch_grupper_immediately=True,
                num_builds=2,
                reset_owners=False,
                ignore_blacklists=True,
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.robot_token,
                description='Scatter tests in branch {}'.format(self.Parameters.merge_tests_branch),
                owner=self.owner,
            ).enqueue()
            self.set_info(
                '<a href={}>BROWSER_MERGE_SCATTER_TESTS launched</a>'.format(get_task_link(scatter_task.id)),
                do_escape=False
            )
            self.Context.scatter_task = scatter_task.id
            raise sdk2.WaitTask(
                [scatter_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        with self.memoize_stage.wait_grupper_launched():
            scatter_task = sdk2.Task.find(id=self.Context.scatter_task).first()
            if scatter_task.status != ctt.Status.SUCCESS:
                raise TaskFailure('Scatter tests failed, cannot go any further')
            grupper_task = sdk2.Task.find(id=scatter_task.Context.grupper_task).first()
            self.set_info(
                '<a href={}>Grupper launched</a>'.format(get_task_link(grupper_task.id)),
                do_escape=False)
            self.Context.grupper_task = grupper_task.id
            targets = {self.Context.grupper_task: 'grupper_addr'}
            raise sdk2.WaitOutput(targets, wait_all=True, timeout=5 * 60)

        grupper_task = sdk2.Task.find(id=self.Context.grupper_task).first()
        if grupper_task.status == ctt.Status.TIMEOUT:
            self.set_info('Grupper timed out, finishing this task')
            return
        if grupper_task.status in ctt.Status.Group.BREAK:
            raise TaskFailure('Grupper did not start, cannot go any further')
        self.Parameters.grupper_addr = grupper_task.Parameters.grupper_addr

        with self.memoize_stage.wait_grupper_finished():
            raise sdk2.WaitTask(
                [grupper_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        if grupper_task.status == ctt.Status.TIMEOUT:
            self.set_info('Grupper timed out, finishing this task')
            return
        if grupper_task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Grupper failed, cannot go any further')

        with self.memoize_stage.create_issues():
            tags = [
                'Merge',
                'Merge{}'.format(self.Parameters.merge_version.split('.')[0]),
                'Merge{}'.format(self.Parameters.merge_version)
            ]
            groups_resource = sdk2.Resource.find(id=grupper_task.Context.groups_resource).first()
            yaml2issues_task = BrowserMergeYamlToIssues.BrowserMergeYamlToIssues(
                self,
                yin_branch=self.Parameters.yin_branch,
                groups_resource=groups_resource,
                verbose=True,
                update_blacklists=False,
                no_dry_run=True,
                test_st_server=False,
                no_wiki=True,
                issues_priority='blocker',
                issues_tags=tags,
                replace_tags=True,
                ticket_template='merge_blocker',
                title_suffix='in merge {}'.format(self.Parameters.merge_version),
                robot_login=self.Parameters.robot_login,
                robot_token=self.Parameters.robot_token,
                description="Create tickets for tests failing in version {}".format(self.Parameters.merge_version),
                owner=self.owner,
            ).enqueue()
            self.set_info(
                '<a href={}>BROWSER_MERGE_YAML_TO_ISSUES launched</a>'.format(get_task_link(yaml2issues_task.id)),
                do_escape=False)
            self.Context.yaml2issues_task = yaml2issues_task.id
            raise sdk2.WaitTask(
                [yaml2issues_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        yaml2issues_task = sdk2.Task.find(id=self.Context.yaml2issues_task).first()
        if yaml2issues_task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Yaml to issues failed, cannot go any further')
