import logging
import shutil
import requests

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc
from sandbox.common.types import client as ctc
from sandbox.projects.common import network
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.merge.grupper import (
    publish_merge_tests, publish_merge_grupper_state,
    BrowserMergeTests, BrowserMergeGrupperState
)
from sandbox.sandboxsdk.environments import NodeJS

GRUPPER_LXC_RESOURCE_ID = 1419202484  # xenial


class BrowserMergeGrupper(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.LINUX_XENIAL
        disk_space = 10 * 1024  # 10GB
        dns = misc.DnsType.DNS64
        environments = [
            NodeJS('10.14.2'),
        ] + RunFromYinTask.Requirements.environments.default

        privileged = True

    class Parameters(RunFromYinTask.Parameters):
        yin_branch = sdk2.parameters.String('Branch in YIN repo to get scripts from',
                                            default='master')

        tests_resource = sdk2.parameters.Resource('Sandbox resource with ungrouped tests',
                                                  required=True, resource_type=BrowserMergeTests)
        state_resource = sdk2.parameters.Resource('Sandbox resource with state tests',
                                                  required=False, resource_type=BrowserMergeGrupperState)
        publish_state = sdk2.parameters.Bool('Publish state resource', default=True)

        _container = sdk2.parameters.Container(
            'Linux container', default_value=GRUPPER_LXC_RESOURCE_ID)

        with sdk2.parameters.Output:
            grupper_addr = sdk2.parameters.String("IP address of grupper")

    class Context(RunFromYinTask.Context):
        groups_resource = None

    @property
    def output_path(self):
        return str(self.path('grupper_output.json'))

    @property
    def state_path(self):
        return str(self.path('grupper_state.json'))

    def on_prepare(self):
        """
        for privileged containers, cloning must be done in on_prepare(),
        after that caches become readonly
        https://wiki.yandex-team.ru/sandbox/cookbook/#privileged-task
        """
        repositories.Stardust.yin(filter_branches=False).clone(
            self.yin_path(), self.Parameters.yin_branch,
            commit=self.Parameters.yin_commit or None)

    @sdk2.header(title='Links')
    def header(self):
        if self.Parameters.grupper_addr:
            return '''
            <p>App is served on <a href="http://[{0}]">{0}</a><br/>
            If unavailable, use <a href="http://grupper-proxy.vla.yp-c.yandex.net/proxy/{0}">proxied link</a></p>
            '''.format(self.Parameters.grupper_addr)
        else:
            return ''

    def on_execute(self):
        logging.debug('Get tests from the resource')
        ungrouped_tests = str(sdk2.ResourceData(self.Parameters.tests_resource).path)

        try:
            logging.debug('Get container address')
            self.Parameters.grupper_addr = network.get_my_ipv6()
            if self.Parameters.state_resource is not None:
                shutil.copyfile(str(sdk2.ResourceData(self.Parameters.state_resource).path),
                                self.state_path)

            logging.debug('Run yin script')
            self.run_yin_script('yin.merge.tools.unnamed_grouping_tool',
                                [
                                    ungrouped_tests,
                                    self.output_path,
                                    self.state_path,
                                    '-vvvv',
                                ],
                                clone_yin=False)
            self.set_info('Terminating the app, publishing the results')
        except Exception:
            logging.exception('FAILED yin.merge.tools.unnamed_grouping_tool')
            raise errors.TaskFailure('FAILED yin.merge.tools.unnamed_grouping_tool')
        finally:
            groups_resource = publish_merge_tests(self, 'Groups and tickets')
            self.Context.groups_resource = groups_resource.id
            if self.Parameters.publish_state:
                publish_merge_grupper_state(self)

    def on_break(self, prev_status, status):
        publish_merge_tests(self, 'Groups and tickets')
        if self.Parameters.publish_state:
            publish_merge_grupper_state(self)

    def timeout_checkpoints(self):
        return [30, 60]

    def on_before_timeout(self, seconds):
        self.set_info('Terminating the app before timeout is reached')
        response = requests.post('http://[{}]/api/terminate'.format(self.Parameters.grupper_addr))
        logging.debug(response.json())
        response.raise_for_status()
