import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc
from sandbox.common.utils import get_task_link
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.merge.grupper import publish_merge_tests
from sandbox.projects.browser.merge.grupper.BrowserMergeGrupper import BrowserMergeGrupper
from sandbox.projects.browser.common.git import repositories


class BrowserMergeScatterTests(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 10 * 1024  # 10GB
        dns = misc.DnsType.DNS64

    class Parameters(RunFromYinTask.Parameters):
        yin_branch = sdk2.parameters.String('Branch in YIN repo to get scripts from',
                                            default='master')

        with sdk2.parameters.Group('Browser repo settings') as repo_settings_group:
            tc_branch = sdk2.parameters.String(
                'Script will ask TeamCity for failed tests in builds on this branch', required=True
            )
            checkout_branch = sdk2.parameters.String(
                'Script will checkout this branch to get configs from it', required=True, default='master'
            )

        with sdk2.parameters.Group('Scatter tests settings') as scatter_settings_group:
            launch_grupper_immediately = sdk2.parameters.Bool('Launch BROWSER_MERGE_GRUPPER asap',
                                                              default=False)
            num_builds = sdk2.parameters.Integer('Look through N latest builds', default=200)
            reset_owners = sdk2.parameters.Bool('Calculate owners again', default=False)
            ignore_blacklists = sdk2.parameters.Bool(
                'Ignore reasons. Ignore --version. Scatter all failed tests.',
                default=False)
            with ignore_blacklists.value[False]:
                version = sdk2.parameters.String(
                    'Chromium version of the merge. Added to the output file. '
                    'Is used to scatter tests ONLY with reasons [version, chromium-<version>]')
            drop_ignored_tests = sdk2.parameters.Bool('Do not include ignored tests in the final report',
                                                      default=False)
            drop_binaries = sdk2.parameters.Bool('Do not include whole failed binaries in the final report',
                                                 default=False)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff('Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String('Vault item with robot token', default='robot-bro-merge_token')

    class Context(RunFromYinTask.Context):
        tests_resource = None
        grupper_task = None

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @property
    def output_path(self):
        return str(self.path('scatter_output.json'))

    def on_execute(self):
        try:
            logging.debug('Checkout browser')
            vcs_root = repositories.Stardust.browser()
            vcs_root.clone(self.browser_path, branch=self.Parameters.checkout_branch)

            args = [
                self.Parameters.tc_branch,
                self.browser_path,
                self.output_path,
                '-n', str(self.Parameters.num_builds),
                '--oauth-login', self.Parameters.robot_login,
                '--oauth-token-env', 'ROBOT_TOKEN',
                '--oauth-no-save'
            ]
            if self.Parameters.reset_owners:
                args.append('--reset-owners')
            if self.Parameters.drop_ignored_tests:
                args.append('--drop-ignored')
            if self.Parameters.drop_binaries:
                args.append('--drop-binaries')
            if self.Parameters.ignore_blacklists:
                args.append('--all')
            else:
                args.extend(['--version', self.Parameters.version])
            logging.info('Args for scatter_tests: %s', args)

            logging.debug('Run yin script')
            try:
                self.run_yin_script('yin.merge.tools.post_merge.scatter_tests',
                                    script_args=args,
                                    extra_env={
                                        'ROBOT_TOKEN': sdk2.Vault.data(self.Parameters.robot_token)
                                    })
            except Exception:
                logging.exception('FAILED yin.merge.tools.post_merge.scatter_tests')
                raise errors.TaskFailure('FAILED yin.merge.tools.post_merge.scatter_tests')
        finally:
            tests_resource = publish_merge_tests(self, 'Tests for {}'.format(self.Parameters.tc_branch))
            self.Context.tests_resource = tests_resource.id

            grupper_task = BrowserMergeGrupper(
                self,
                yin_branch=self.Parameters.yin_branch,
                tests_resource=tests_resource,
                state_resource=None,
                publish_state=True,
                description="Grouping tests in branch {co_b}. Using {n} builds from branch {tc_b}".format(
                    co_b=self.Parameters.checkout_branch,
                    n=self.Parameters.num_builds,
                    tc_b=self.Parameters.tc_branch
                ),
                owner=self.owner,
                kill_timeout=self.Parameters.kill_timeout,
            )
            self.Context.grupper_task = grupper_task.id

            if self.Parameters.launch_grupper_immediately:
                self.set_info(
                    '<a href={}>Grupper launched</a>'.format(get_task_link(grupper_task.id)),
                    do_escape=False)
                grupper_task.enqueue()
            else:
                self.set_info(
                    'Would you like to <a href={}>launch grupper?</a>'.format(get_task_link(grupper_task.id)),
                    do_escape=False)

    def on_break(self, prev_status, status):
        publish_merge_tests(self, 'Tests for {}'.format(self.Parameters.tc_branch))
