import enum
import logging
import os
import subprocess
import time
import yaml

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import misc
from sandbox.common.utils import singleton_property, get_task_link
from sandbox.projects.browser.common.bitbucket import BitBucket, DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.RunFromYinTask import RunFromYinTask
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.browser.merge.grupper import BrowserMergeTests, publish_merge_tests

ST_TEST_BASE_API_URL = 'https://st-api.test.yandex-team.ru'
DEFAULT_ISSUES_AUTHOR = 'robot-bro-merge'


class TicketProfile(enum.Enum):
    after_merge = 1
    flakiness = 2
    master_failures = 3
    merge_blocker = 4
    update_release = 5
    custom_tickets = 6


class BrowserMergeYaml2IssuesErrors(sdk2.Resource):
    ttl = "inf"


class BrowserMergeYamlToIssues(RunFromYinTask):
    class Requirements(RunFromYinTask.Requirements):
        disk_space = 3 * 1024  # 3GB
        dns = misc.DnsType.DNS64

    class Parameters(RunFromYinTask.Parameters):
        with sdk2.parameters.Group('General settings') as general_settings_group:
            yin_branch = sdk2.parameters.String('Branch in YIN repo to get scripts from',
                                                default='master')
            groups_resource = sdk2.parameters.Resource(
                'Sandbox resource with grouped tests', required=True,
                resource_type=BrowserMergeTests)
            verbose = sdk2.parameters.Bool('Verbose logging level', default=True)

        with sdk2.parameters.Group('Update blacklists settings') as blacklist_settings_group:
            update_blacklists = sdk2.parameters.Bool('Update blacklists with issues', default=False)
            with update_blacklists.value[True]:
                browser_branch = sdk2.parameters.String('Branch in browser repo', required=True)
                reviewers = sdk2.parameters.List('Reviewers',
                                                 required=True,
                                                 default=['hpotter', 'anata-m'])
                take_version_from_yaml = sdk2.parameters.Bool(
                    "Use yaml's 'version' field and replace tests with this value with tracker issues",
                    default=True)
                with take_version_from_yaml.value[False]:
                    replace_reason = sdk2.parameters.String(
                        "Use custom string and replace tests with this value with tracker issues",
                        required=True)

        with sdk2.parameters.Group('Publishing settings') as publ_settings_group:
            no_dry_run = sdk2.parameters.Bool('--no-dry-run. Create issues in Tracker',
                                              default=False)
            with no_dry_run.value[True]:
                test_st_server = sdk2.parameters.Bool(
                    'Use test Tracker', default=False,
                    description='Use test Tracker server url: {url}'.format(url=ST_TEST_BASE_API_URL))
            no_wiki = sdk2.parameters.Bool('Do not update wiki page', default=False)

        with sdk2.parameters.Group('New issues settings') as issues_settings_group:
            issues_tags = sdk2.parameters.List('Specify tags', default=[])
            replace_tags = sdk2.parameters.Bool('Replace template tags with ones from parameters',
                                                default=False)
            issues_components = sdk2.parameters.List('Specify components', default=[])
            replace_components = sdk2.parameters.Bool('Replace template components with ones from parameters',
                                                      default=False)
            title_suffix = sdk2.parameters.String('Specify title suffix')
            add_description = sdk2.parameters.String('This comment will be added at the beginning of each ticket',
                                                     default='')
            issues_priority = sdk2.parameters.String('Specify priority',
                                                     choices=[('None', None),
                                                              ('critical', 'critical'),
                                                              ('blocker', 'blocker')],
                                                     default=None)
            fix_version = sdk2.parameters.String('Specify fix version')
            ticket_template = sdk2.parameters.String(
                'Tracker ticket template',
                choices=[(t.name, t.name) for t in TicketProfile],
                default=TicketProfile.after_merge.name,
                description='Determines how issues are pre-filled '
                            '(issue tags, components, summary, etc)')
            issues_author = sdk2.parameters.String('Issues author', default=DEFAULT_ISSUES_AUTHOR)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            robot_login = sdk2.parameters.Staff('Robot login', default='robot-bro-merge')
            robot_token = sdk2.parameters.String('Vault item with robot token', default='robot-bro-merge_token')
            with update_blacklists.value[True]:
                robot_ssh_key_vault = sdk2.parameters.String('Vault item with ssh key for bitbucket',
                                                             default='robot-bro-merge_ssh_key',
                                                             required=True)

    class Context(RunFromYinTask.Context):
        groups_with_issues_resource = None

    @property
    def output_path(self):
        return str(self.path('yaml2issues_output.json'))

    @property
    def error_path(self):
        return str(self.path('yaml2issues_errors.json'))

    @property
    def browser_path(self):
        return str(self.path('browser'))

    @singleton_property
    def git(self):
        return GitCli(self.browser_path, config={
            'user.name': self.Parameters.robot_login,
            'user.email': '{}@yandex-team.ru'.format(self.Parameters.robot_login)
        })

    @singleton_property
    def bb(self):
        return BitBucket(DEFAULT_BITBUCKET_URL, 'x-oauth-token', sdk2.Vault.data(self.Parameters.robot_token))

    def on_execute(self):
        with self.memoize_stage.yaml_to_issues:
            try:
                yaml_to_issues_args = [
                    str(sdk2.ResourceData(self.Parameters.groups_resource).path),
                    self.output_path,
                    '--ticket-template', self.Parameters.ticket_template,
                    '--issues-author', self.Parameters.issues_author,
                    '--error-output', self.error_path,
                    '--oauth-login', self.Parameters.robot_login,
                    '--oauth-token-env', 'ROBOT_TOKEN',
                    '--oauth-no-save'
                ]
                if self.Parameters.title_suffix:
                    yaml_to_issues_args.append('--title-suffix')
                    yaml_to_issues_args.append(self.Parameters.title_suffix)
                else:
                    self.set_info("title_suffix not specified; will use script's default")
                if self.Parameters.issues_priority:
                    yaml_to_issues_args.append('--priority')
                    yaml_to_issues_args.append(self.Parameters.issues_priority)
                else:
                    self.set_info(
                        'issues_priority not specified; will use default from template "{}"'.format(
                            self.Parameters.ticket_template
                        ))

                if self.Parameters.verbose:
                    yaml_to_issues_args.append('--verbose')
                if self.Parameters.no_dry_run:
                    yaml_to_issues_args.append('--no-dry-run')
                    if self.Parameters.test_st_server:
                        yaml_to_issues_args.append('--test-startrek-server')
                if self.Parameters.no_wiki:
                    yaml_to_issues_args.append('--nowiki')
                if self.Parameters.fix_version:
                    yaml_to_issues_args.append('--fix-version')
                    yaml_to_issues_args.append(self.Parameters.fix_version)

                for tag in self.Parameters.issues_tags:
                    yaml_to_issues_args.append('--tag')
                    yaml_to_issues_args.append(tag)
                if self.Parameters.replace_tags:
                    yaml_to_issues_args.append('--replace-tags')
                for component in self.Parameters.issues_components:
                    yaml_to_issues_args.append('--component')
                    yaml_to_issues_args.append(component)
                if self.Parameters.replace_components:
                    yaml_to_issues_args.append('--replace-components')
                if self.Parameters.add_description:
                    yaml_to_issues_args.append('--add-description')
                    yaml_to_issues_args.append(self.Parameters.add_description)
                logging.info('Args for yaml2issues: %s', yaml_to_issues_args)

                logging.debug('Run yin script yaml2issues')
                try:
                    self.run_yin_script('yin.merge.tools.post_merge.yaml2issues',
                                        script_args=yaml_to_issues_args,
                                        extra_env={
                                            'ROBOT_TOKEN': sdk2.Vault.data(self.Parameters.robot_token)
                                        })
                except Exception:
                    logging.exception('FAILED yin.merge.tools.post_merge.yaml2issues')
                    raise errors.TaskFailure('FAILED yin.merge.tools.post_merge.yaml2issues')
            finally:
                groups_with_issues_resource = publish_merge_tests(self, 'YAML with issues')
                self.Context.groups_with_issues_resource = groups_with_issues_resource.id
                if os.path.exists(self.error_path):
                    errors_resource = BrowserMergeYaml2IssuesErrors(
                        self, 'Startrek errors', self.error_path
                    )
                    sdk2.ResourceData(errors_resource).ready()
                    self.set_info(
                        'Some issues were not created - see <a href="{0}">errors list</a> for details'.format(
                            errors_resource.http_proxy
                        ),
                        do_escape=False
                    )

        if self.Parameters.update_blacklists:
            pr_branch = 'wp/robots/issues_to_blacklist/{}'.format(int(time.time() * 1000))
            if self.Parameters.take_version_from_yaml:
                with open(str(self.output_path), 'r') as f:
                    tmp = yaml.safe_load(f)
                    reason_to_replace = tmp['version']
            else:
                reason_to_replace = self.Parameters.replace_reason

            self.set_info('Will replace reason {} with tracker issues'.format(reason_to_replace))
            try:
                logging.info('Checkout browser branch %s', self.Parameters.browser_branch)
                vcs_root = repositories.Stardust.browser()
                vcs_root.clone(self.browser_path, self.Parameters.browser_branch)

                self.git.checkout('-f', '-B', pr_branch, self.Parameters.browser_branch)
            except subprocess.CalledProcessError:
                raise errors.TemporaryError('Checkout error')

            issues_to_bl_args = [
                self.output_path,
                self.browser_path,
                reason_to_replace
            ]
            logging.debug('Run yin script issues2blacklists')
            self.run_yin_script('yin.merge.tools.post_merge.issues2blacklists',
                                script_args=issues_to_bl_args,
                                extra_env={'ROBOT_TOKEN': sdk2.Vault.data(self.Parameters.robot_token)},
                                clone_yin=False)  # already cloned

            try:
                self.git.commit(
                    '-a', '-m',
                    'Replacing reason {} with issues. Committed by sandbox task: {}.'.format(
                        self.Parameters.replace_reason,
                        get_task_link(self.id)
                    ),
                )
                with sdk2.ssh.Key(self, self.Parameters.robot_ssh_key_vault, None):
                    self.git.push('origin', '{}:refs/heads/{}'.format(pr_branch, pr_branch))
            except subprocess.CalledProcessError:
                raise errors.TaskFailure('Failed to commit or push changes')

            pr = self.bb.create_pr('stardust', 'browser',
                                   'Replacing reason {} with issues.'.format(self.Parameters.replace_reason),
                                   description='Created from sandbox task {}'.format(get_task_link(self.id)),
                                   from_ref=pr_branch, to_ref=self.Parameters.browser_branch,
                                   reviewers=self.Parameters.reviewers)
            self.set_info('Created <a href="{}">pull request</a>'.format(pr.web_url),
                          do_escape=False)
