import logging
import time

from sandbox import sdk2
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin
from sandbox.projects.browser.common.contextmanagers import ExitStack, TempEnvironment
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.metrics.common import BrowserMeasureGitTime


INCREMENTAL_FETCH_HTTPS_VS_SSH_CHART_FEED_ID = '6v7i9w61atpgy'
INCREMENTAL_FETCH_BY_SIMULTANEITY_CHART_FEED_ID = 'j8kvffak8m1qb'
FULL_FETCH_HTTPS_VS_SSH_CHART_FEED_ID = 'd28gec65y9fq5'
FULL_FETCH_BY_SIMULTANEITY_CHART_FEED_ID = 'qflrkgbxh7pgi'


class BrowserMeasureGitFetchTime(BrowserMeasureGitTime, LinuxBinaryTaskMixin, sdk2.Task):
    class Parameters(BrowserMeasureGitTime.Parameters):
        is_simultaneous = sdk2.parameters.Bool('There is another task run at the same time', default=False)

        with sdk2.parameters.Group('Branch settings') as branch_settings:
            full_fetch_branch = sdk2.parameters.String(
                'First branch for "full" git-fetch (e.g. \'metrics/fetch/full\', \'metrics-fetch-from-branch\').'
                ' First of all, task fetches this branch',
                required=True
            )
            incremental_fetch_branch = sdk2.parameters.String(
                'Second branch for "incremental" git-fetch'
                ' (e.g. \'metrics/fetch/incremental\', \'metrics-fetch-to-branch\').'
                ' Secondarily, task fetches this branch.'
                ' first fetch branch\'s HEAD commit has to be ancestor commit of second fetch branch\'s HEAD commit',
                required=True
            )

    @property
    def yt_table_path_template(self):
        return '//home/browser/infra/statistics/bitbucket_git/fetch/{date}'

    def get_fetch_time_seconds(self, branch, transfer_protocol, ssh_key_optional):
        """
        :param branch: branch name to git-fetch. For the purpose of this task, it can be branch only not tag.
        :type branch: str
        :param transfer_protocol: either 'https' or 'ssh'
        :type transfer_protocol: str
        :param ssh_key_optional: pass None if no SSH involved
        :type ssh_key_optional: sdk2.ssh.Key|None
        :rtype: float
        """
        assert transfer_protocol != 'ssh' or ssh_key_optional is not None, 'Specify SSH key or do not use SSH'

        time_before_fetch = time.time()

        with ExitStack() as stack:
            if transfer_protocol == 'ssh':
                stack.enter_context(ssh_key_optional)
            with TempEnvironment() as temp_env:
                temp_env.set_var('GIT_TRACE', '1')
                self.git_cli.fetch('origin', branch)

        time_after_fetch = time.time()

        time_taken_in_seconds = time_after_fetch - time_before_fetch
        logging.debug('Took %.3f seconds', time_taken_in_seconds)
        return time_taken_in_seconds

    @property
    def yt_table_attributes_json(self):
        """
        :return: JSON with scheme for data that's going to be uploaded to YT
        :rtype: dict[str, Any]
        """
        return {
            'unique_keys': True,
            'schema': {
                '$value': [
                    {
                        'type': 'float',
                        'name': 'execute_timestamp',
                        'required': True
                    },
                    # Branch/commit information
                    {
                        'type': 'string',
                        'name': 'fetch_branch',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'fetch_commit',
                        'required': True
                    },
                    # Launch parameters
                    {
                        'type': 'string',
                        'name': 'fetch_type',
                        'required': True
                    },
                    {
                        'type': 'boolean',
                        'name': 'is_simultaneous',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'project_id',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'repository_id',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'transfer_protocol',
                        'required': True
                    },
                    # Result
                    {
                        'type': 'float',
                        'name': 'fetch_time_seconds',
                        'required': True
                    },
                ],
                '$attributes': {
                    'strict': True,
                }
            }
        }

    def build_statistics_for_yt(self, fetch_time_seconds, fetch_type, execute_timestamp, branch, commit):
        """
        :param fetch_time_seconds: time taken for git-fetch
        :type fetch_time_seconds: float
        :param fetch_type: either 'full' or 'incremental'
        :type fetch_type: str
        :type execute_timestamp: float
        :type branch: str
        :type commit: str
        :rtype: dict[str, Any]
        """
        assert fetch_type in {'full', 'incremental'}, 'fetch_type can be either full or incremental'
        return {
            'execute_timestamp': execute_timestamp,

            'fetch_branch': branch,
            'fetch_commit': commit,

            'fetch_type': fetch_type,
            'is_simultaneous': self.Parameters.is_simultaneous,
            'project_id': self.PROJECT_ID,
            'repository_id': self.Parameters.repository_id,
            'transfer_protocol': self.Parameters.transfer_protocol,

            'fetch_time_seconds': fetch_time_seconds,
        }

    def prepare_local_repo(self):
        """
        :rtype: None
        """
        self.local_repo_absolute_path.mkdir()
        self.git_cli.init(str(self.local_repo_absolute_path))

        # Get repository address to fetch.
        if self.Parameters.transfer_protocol == 'https':
            repository_address = repositories.get_https_git_url(
                repo=self.Parameters.repository_id, project=self.PROJECT_ID
            )
        elif self.Parameters.transfer_protocol == 'ssh':
            repository_address = repositories.get_ssh_git_url(
                repo=self.Parameters.repository_id, project=self.PROJECT_ID
            )
        else:
            assert False, 'Not reachable'

        self.git_cli.remote('add', 'origin', repository_address)

    def get_measured_git_commands(self):
        # Git-fetch full_fetch_branch and then incremental_fetch_branch, measure time of git-fetch executing.
        full_fetch_time_seconds = self.get_fetch_time_seconds(
            branch=self.Parameters.full_fetch_branch,
            transfer_protocol=self.Parameters.transfer_protocol,
            ssh_key_optional=self.ssh_key_optional
        )
        incremental_fetch_time_seconds = self.get_fetch_time_seconds(
            branch=self.Parameters.incremental_fetch_branch,
            transfer_protocol=self.Parameters.transfer_protocol,
            ssh_key_optional=self.ssh_key_optional
        )

        # Get commits' ids.
        full_fetch_commit = self.get_commit_by_branch(
            branch=self.Parameters.full_fetch_branch,
            ssh_key_optional=self.ssh_key_optional,
            transfer_protocol=self.Parameters.transfer_protocol
        )
        incremental_fetch_commit = self.get_commit_by_branch(
            branch=self.Parameters.incremental_fetch_branch,
            ssh_key_optional=self.ssh_key_optional,
            transfer_protocol=self.Parameters.transfer_protocol
        )

        # Get statistics in format suitable to upload to YT.
        full_fetch_statistics = self.build_statistics_for_yt(
            fetch_time_seconds=full_fetch_time_seconds,
            fetch_type='full',
            # TODO (BYIN-14493): switch to .timestamp() after dropping python2
            execute_timestamp=self.created_with_timezone_timestamp,
            branch=self.Parameters.full_fetch_branch,
            commit=full_fetch_commit,
        )
        incremental_fetch_statistics = self.build_statistics_for_yt(
            fetch_time_seconds=incremental_fetch_time_seconds,
            fetch_type='incremental',
            # TODO (BYIN-14493): switch to .timestamp() after dropping python2
            execute_timestamp=self.created_with_timezone_timestamp,
            branch=self.Parameters.incremental_fetch_branch,
            commit=incremental_fetch_commit
        )

        return [full_fetch_statistics, incremental_fetch_statistics]

    @property
    def feed_ids(self):
        """
        :return: feed IDs of corresponding charts
        :rtype: list[str]
        """
        return [
            INCREMENTAL_FETCH_HTTPS_VS_SSH_CHART_FEED_ID,
            INCREMENTAL_FETCH_BY_SIMULTANEITY_CHART_FEED_ID,
            FULL_FETCH_HTTPS_VS_SSH_CHART_FEED_ID,
            FULL_FETCH_BY_SIMULTANEITY_CHART_FEED_ID,
        ]
